/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.application.db;

import org.eclipse.equinox.app.IApplication;
import org.eclipse.equinox.app.IApplicationContext;
import org.simantics.application.arguments.SimanticsArguments;
import org.simantics.application.template.TemplateDatabaseApplication;
import org.simantics.db.Session;

/**
 * A headless (non-ui) application base class for one-shot connection to and
 * usage of a Simantics database.
 * 
 * <p>
 * The simplest way to use this application class is to override the
 * {@link #afterConnect(DatabaseHandle)} method to do your own stuff with the
 * database in question. Use {@link #getSession()} to get a hold of the database
 * session.
 * </p>
 * 
 * <p>
 * The application expects at least the {@link SimanticsArguments#SERVER} and
 * supports the {@link SimanticsArguments#CHECKOUT}.
 * </p>
 * 
 * @see TemplateDatabaseApplication for a usage example
 */
public abstract class HeadlessDatabaseApplication implements IApplication {

    private DatabaseHandle database;

    @Override
    public Object start(IApplicationContext context) throws Exception {
        try {
            beforeConnect(context);
            connectToServer(context);
            afterConnect();
            
            return IApplication.EXIT_OK;
        } finally {
            if (database != null) {
                try {
                    database.dispose();
                } catch (Throwable t) {
                    t.printStackTrace();
                }
            }
        }
    }

    @Override
    public void stop() {
        // TODO: force the application to stop
    }

    protected IDatabaseHandle connectToServer(IApplicationContext context) throws Exception {
        String[] args = (String[]) context.getArguments().get(IApplicationContext.APPLICATION_ARGS);
        database = new DatabaseHandle(args, deleteLocalCopyAtExit());
        return database;
    }

    protected Session getSession() {
        if (database == null)
            throw new IllegalStateException("null database session");
        return database.getSession();
    }

    /**
     * Override this method to perform actions <em>before</em> the application
     * attempts to connect to the database specified by the command line
     * arguments.
     * 
     * @param context the application startup context
     */
    protected void beforeConnect(IApplicationContext context) {
    }

    /**
     * Override this method to perform actions <em>after</em> the application
     * has connected to the database specified by the command line arguments.
     */
    protected void afterConnect() {
    }

    protected boolean deleteLocalCopyAtExit() {
        return false;
    }

}
