/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.annotation.ui.wizard;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.simantics.NameLabelMode;
import org.simantics.NameLabelUtil;
import org.simantics.annotation.ontology.AnnotationResource;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.NamedResource;
import org.simantics.db.common.request.PossibleIndexRoot;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.SelectionHints;
import org.simantics.db.layer0.adapter.Instances;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.layer0.Layer0;
import org.simantics.utils.strings.AlphanumComparator;
import org.simantics.utils.ui.ISelectionUtils;

/**
 * @author Tuukka Lehtonen
 * @author Teemu Mtsniemi
 */
public class AnnotationTypeExportPage extends WizardPage {

    ExportPlan          exportModel;
    CCombo              model;
    CCombo              exportLocation;

    List<NamedResource> models = Collections.emptyList();
    private Button      overwrite;

    protected AnnotationTypeExportPage(ExportPlan model) {
        super("Export Annotation Type", "Define Export Location", null);
        this.exportModel = model;
    }

    @Override
    public void createControl(Composite parent) {
        Composite container = new Composite(parent, SWT.NONE);
        {
            GridLayout layout = new GridLayout();
            layout.horizontalSpacing = 20;
            layout.verticalSpacing = 10;
            layout.numColumns = 3;
            container.setLayout(layout);
        }

        new Label(container, SWT.NONE).setText("Exported &annotation type:");
        model = new CCombo(container, SWT.BORDER);
        {
            model.setEditable(false);
            model.setText("");
            model.setToolTipText("Selects the annotation type to export.");
            GridDataFactory.fillDefaults().grab(true, false).span(2, 1).applyTo(model);
            model.addModifyListener(new ModifyListener(){
                @Override
                public void modifyText(ModifyEvent e) {
                    validatePage();
                }
            });
        }

        new Label(container, SWT.NONE).setText("&Target file:");
        exportLocation = new CCombo(container, SWT.BORDER);
        {
            exportLocation.setText("");
            GridDataFactory.fillDefaults().grab(true, false).span(1, 1).applyTo(exportLocation);
            exportLocation.addModifyListener(new ModifyListener(){
                @Override
                public void modifyText(ModifyEvent e) {
                    validatePage();
                }
            });
        }
        Button browseFileButton = new Button(container, SWT.PUSH);
        {
            browseFileButton.setText("Browse...");
            browseFileButton.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, false, false));
            browseFileButton.addSelectionListener(new SelectionAdapter() {
                @Override
                public void widgetSelected(SelectionEvent e) {
                    FileDialog dialog = new FileDialog(getShell(), SWT.SAVE);
                    dialog.setText("Choose Export Target File");
                    String loc = exportLocation.getText();
                    dialog.setFilterPath(loc);
                    dialog.setFilterExtensions(new String[] { "*.annotationType" });
                    dialog.setFilterNames(new String[] { "Annotation Type (*.annotationType)" });
                    dialog.setOverwrite(false);
                    String file = dialog.open();
                    if (file == null)
                        return;
                    exportLocation.setText(file);
                    validatePage();
                }
            });
        }

        Label horizRule = new Label(container, SWT.BORDER);
        GridDataFactory.fillDefaults().hint(SWT.DEFAULT, 0).grab(true, false).span(3, 1).applyTo(horizRule);

        overwrite = new Button(container, SWT.CHECK);
        overwrite.setText("&Overwrite existing files without warning");
        overwrite.setSelection(exportModel.overwrite);
        GridDataFactory.fillDefaults().grab(true, false).span(3, 1).applyTo(overwrite);
        overwrite.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                validatePage();
            }
        });

        try {
            initializeData();
        } catch (DatabaseException e) {
            e.printStackTrace();
        }

        setControl(container);
        validatePage();
    }

    private void initializeData() throws DatabaseException {
        final List<Resource> resources = ISelectionUtils.getPossibleKeys(exportModel.selection, SelectionHints.KEY_MAIN, Resource.class);

        models = exportModel.sessionContext.getSession().syncRequest(new UniqueRead<List<NamedResource>>() {
            AnnotationResource ANNO;
            Layer0 L0;
            NameLabelMode mode;

            @Override
            public List<NamedResource> perform(ReadGraph graph) throws DatabaseException {
                ANNO = AnnotationResource.getInstance(graph);
                L0 = Layer0.getInstance(exportModel.sessionContext.getSession());
                mode = NameLabelMode.NAME; //NameLabelUtil.getNameLabelMode(graph);

                Map<String, NamedResource> result = new TreeMap<String, NamedResource>(AlphanumComparator.CASE_INSENSITIVE_COMPARATOR);
                for (Resource r : resources)
                    process(graph, r, result);

                if (result.isEmpty()) {
                    // Look for all annotation types in the index roots contained by the selected resources.
                    Set<Resource> visited = new HashSet<Resource>();
                    Instances types = graph.adapt(ANNO.AnnotationType, Instances.class);
                    for (Resource r : resources) {
                        Resource indexRoot = graph.syncRequest(new PossibleIndexRoot(r));
                        if (indexRoot == null || !visited.add(indexRoot))
                            continue;
                        for (Resource at : types.find(graph, indexRoot)) {
                            Resource atRoot = graph.syncRequest(new PossibleIndexRoot(at));
                            if (indexRoot.equals(atRoot))
                                process(graph, at, result);
                        }
                    }
                }

                return new ArrayList<NamedResource>( result.values() );
            }

            private NamedResource process(ReadGraph graph, Resource r, Map<String, NamedResource> result) throws DatabaseException {
                NamedResource nr = process(graph, r);
                if (nr != null)
                    result.put(nr.getName(), nr);
                return nr;
            }

            private NamedResource process(ReadGraph graph, Resource r) throws DatabaseException {
                if (!graph.isInstanceOf(r, ANNO.AnnotationType))
                    return null;
                Resource property = graph.getPossibleObject(r, L0.HasRange_Inverse);
                String name = pathName(graph, r, property != null ? property : r);
                return new NamedResource(name, r);
            }

            private String pathName(ReadGraph graph, Resource r, Resource namedEntity) throws DatabaseException {
                String name = graph.getRelatedValue(namedEntity, L0.HasName, Bindings.STRING);

                Resource indexRoot = graph.syncRequest(new PossibleIndexRoot(r));
                if (indexRoot == null)
                    return name;
                Variable rootV = Variables.getPossibleVariable(graph, indexRoot);
                Variable v = Variables.getPossibleVariable(graph, r);
                if (v == null || rootV == null)
                    return name;

                List<Variable> path = Variables.getPath(graph, rootV, v);
                path.add(0, rootV);
                StringBuilder sb = new StringBuilder();
                for (Variable p : path) {
                    String segmentName = NameLabelUtil.modalName(graph, p, mode);
                    sb.append(segmentName).append(" / ");
                }
                sb.append(name);
                return sb.toString();
            }
        });

        if (!models.isEmpty())
            exportModel.model = models.get(0);

        // Populate combo boxes
        int i = 0;
        for (NamedResource m : models) {
            model.add(m.getName());
            model.setData(String.valueOf(i), m);
            if (m.equals(exportModel.model))
                model.select(i);
            ++i;
        }

        for (String path : exportModel.recentLocations) {
            exportLocation.add(path);
        }
        if (exportLocation.getItemCount() > 0)
            exportLocation.select(0);
    }

    void validatePage() {
        if (exportModel.model == null) {
            setMessage("Select model to export from.");
            setErrorMessage(null);
            setPageComplete(false);
            return;
        }

        String exportLoc = exportLocation.getText();
        if (exportLoc.isEmpty()) {
            setMessage("Select target file.");
            setErrorMessage(null);
            setPageComplete(false);
            return;
        }
        if (!exportLoc.endsWith(".annotationType"))
            exportLoc += exportLoc.endsWith(".") ? "annotationType" : ".annotationType";
        File file = new File(exportLoc);
        if (file.isDirectory()) {
            setErrorMessage("The target is a directory.");
            setPageComplete(false);
            return;
        }
        File parent = file.getParentFile();
        if (parent == null || !parent.isDirectory()) {
            setErrorMessage("The target directory does not exist.");
            setPageComplete(false);
            return;
        }
        exportModel.exportLocation = file;
        exportModel.overwrite = overwrite.getSelection();

        setErrorMessage(null);
        setMessage("Export annotation type to " + exportLoc + ".");
        setPageComplete(true);
    }

}
