/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.tests;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotSame;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertSame;

import java.io.Serializable;
import java.util.AbstractCollection;
import java.util.AbstractList;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.junit.Before;
import org.junit.Test;
import org.simantics.databoard.util.ArrayUtils;

public class TestArrayUtils {

	@Before
	public void setUp() {
	}
	
	@Test
	public void testConcatenate1() {
		Integer[] a = new Integer[] { 1, 2, 3 };
		Integer[] b = new Integer[] { 4, 5 };
		Integer[] c = ArrayUtils.concatenate(a, b);
		assertEquals(5, c.length);
		for (int i = 0; i < c.length; i++)
			assertEquals(i+1, (int)c[i]);
		assertSame(a[0], c[0]);
		assertSame(b[0], c[3]);
	}
	
	@Test
	public void testConcatenateEmptyA() {
		Integer[] a = new Integer[] { };
		Integer[] b = new Integer[] { 4, 5 };
		Integer[] c = ArrayUtils.concatenate(a, b);
		assertArrayEquals(b, c);
		assertNotSame(b, c);
		assertSame(b[0], c[0]);
		assertSame(b[1], c[1]);
	}

	@Test
	public void testConcatenateEmptyB() {
		Integer[] a = new Integer[] { 1, 2 };
		Integer[] b = new Integer[] { };
		Integer[] c = ArrayUtils.concatenate(a, b);
		assertArrayEquals(a, c);
		assertNotSame(a, c);
		assertSame(a[0], c[0]);
		assertSame(a[1], c[1]);
	}

	@Test
	public void testConcatenateEmptyBoth() {
		Integer[] a = new Integer[] { };
		Integer[] b = new Integer[] { };
		Integer[] c = ArrayUtils.concatenate(a, b);
		assertEquals(0,  c.length);
		assertNotSame(a, c);
		assertNotSame(b, c);
	}
	
	@Test
	public void testConcatenateMixedTypes() {
		Integer[] a = new Integer[] { 1 };
		Double[] b = new Double[] { 2.0 };
		Number[] c = ArrayUtils.concatenate(a, b);
		assertSame(Number[].class, c.getClass());
		assertSame(a[0], c[0]);
		assertSame(b[0], c[1]);
	}

	@Test
	public void testConcatenateMixedTypes2() {
		Integer[] a = new Integer[] { 1 };
		Double[] b = new Double[] { 2.0 };
		Object[] c = ArrayUtils.concatenate(a, b);
		assertSame(c.getClass(), Number[].class);
		assertSame(a[0], c[0]);
		assertSame(b[0], c[1]);
	}	
	
	@Test
	public void testConcatenateMixedTypes3() {
		Integer[] a = new Integer[] { 1 };
		Double[] b = new Double[] { 2.0 };
		Object[] c = ArrayUtils.concatenate(a, b, Object.class);
		assertSame(c.getClass(), Object[].class);
		assertSame(a[0], c[0]);
		assertSame(b[0], c[1]);
	}	
	
	@Test(expected = ClassCastException.class)
	public void testConcatenateMixedTypes4() {
		Integer[] a = new Integer[] { 1 };
		Double[] b = new Double[] { 2.0 };
		@SuppressWarnings("unused")
		Number[] c = (Number[])ArrayUtils.concatenate(a, b, Object.class);
	}
	
	@Test
	public void testConcatenateMixedTypes5() {
		Integer[] a = new Integer[] { 1 };
		String[] b = new String[] { "foo" };
		Object[] c = ArrayUtils.concatenate(a, b);
		assertSame(Object[].class, c.getClass());
		assertEquals(2, c.length);
	}
	
	@Test
	public void testConcatenateMixedTypes6() {
		Object[] a = new CharSequence[] { "bar" };
		Object[] b = new String[] { "foo" };
		Object[] c = ArrayUtils.concatenate(a, b);
		assertSame(CharSequence[].class, c.getClass());
		assertEquals(2, c.length);
	}
	
	@Test
	public void testConcatenateMixedTypes7() {
		Object[] a = new String[] { "bar" };
		Object[] b = new CharSequence[] { "foo" };
		Object[] c = ArrayUtils.concatenate(a, b);
		assertSame(CharSequence[].class, c.getClass());
		assertEquals(2, c.length);
	}
	
	@Test
	public void testConcatenateLengthGt() {
		Integer[] a = new Integer[] { 1, 2, 3 };
		Integer[] b = new Integer[] { 4, 5 };
		Integer[] c = ArrayUtils.concatenate(a, b, 8);
		assertEquals(8, c.length);
		for (int i = 0; i < 5; i++)
			assertEquals(i+1, (int)c[i]);
		for (int i = 5; i < 8; i++)
			assertNull(c[i]);
		assertSame(a[0], c[0]);
		assertSame(b[0], c[3]);
	}

	@Test
	public void testConcatenateLengthLtB() {
		Integer[] a = new Integer[] { 1, 2, 3 };
		Integer[] b = new Integer[] { 4, 5 };
		Integer[] c = ArrayUtils.concatenate(a, b, 4);
		assertEquals(4, c.length);
		for (int i = 0; i < 4; i++)
			assertEquals(i+1, (int)c[i]);
		assertSame(a[0], c[0]);
		assertSame(b[0], c[3]);
	}

	@Test
	public void testConcatenateLengthLtA() {
		Integer[] a = new Integer[] { 1, 2, 3 };
		Integer[] b = new Integer[] { 4, 5 };
		Integer[] c = ArrayUtils.concatenate(a, b, 2);
		assertEquals(2, c.length);
		for (int i = 0; i < 2; i++)
			assertEquals(i+1, (int)c[i]);
		assertSame(a[0], c[0]);
	}
	
	@Test
	public void testConcatenateLengthZero() {
		Integer[] a = new Integer[] { 1, 2, 3 };
		Integer[] b = new Integer[] { 4, 5 };
		Integer[] c = ArrayUtils.concatenate(a, b, 0);
		assertEquals(0, c.length);
	}
	
	@Test
	public void testConcatenateLengthZeroEmptyA() {
		Integer[] a = new Integer[] { };
		Integer[] b = new Integer[] { 4, 5 };
		Integer[] c = ArrayUtils.concatenate(a, b, 0);
		assertEquals(0, c.length);
	}
	
	@Test
	public void testConcatenateLengthEmptyA() {
		Integer[] a = new Integer[] { };
		Integer[] b = new Integer[] { 4, 5 };
		Integer[] c = ArrayUtils.concatenate(a, b, 2);
		assertEquals(2, c.length);
		assertArrayEquals(b,  c);
		assertNotSame(b, c);
	}
	
	@Test
	public void testConcatenateLengthEmptyB() {
		Integer[] a = new Integer[] { 1, 2, 3 };
		Integer[] b = new Integer[] {  };
		Integer[] c = ArrayUtils.concatenate(a, b, 3);
		assertEquals(3, c.length);
		assertArrayEquals(a,  c);
		assertNotSame(a, c);
	}

	@Test
	public void testAppend1() {
		Integer[] a = new Integer[] { 1, 2, 3 };
		Integer[] c = ArrayUtils.append(a, 4, 5);
		assertEquals(5, c.length);
		for (int i = 0; i < c.length; i++)
			assertEquals(i+1, (int)c[i]);
		assertSame(a[0], c[0]);
	}
	
	@Test
	public void testAppendEmptyA() {
		Integer[] a = new Integer[] { };
		Integer b = 4;
		Integer[] c = ArrayUtils.append(a, b, 5);
		assertEquals(2, c.length);
		assertSame(b, c[0]);
		assertEquals(5, (int)c[1]);
	}

	@Test
	public void testAppendEmptyB() {
		Integer[] a = new Integer[] { 1, 2 };
		Integer[] c = ArrayUtils.append(a);
		assertArrayEquals(a, c);
		assertNotSame(a, c);
		assertSame(a[0], c[0]);
		assertSame(a[1], c[1]);
	}

	@Test
	public void testAppendEmptyBoth() {
		Integer[] a = new Integer[] { };
		Integer[] c = ArrayUtils.append(a);
		assertEquals(0,  c.length);
		assertNotSame(a, c);
	}
	
	@Test
	public void testAppendMixedTypes() {
		Integer[] a = new Integer[] { 1 };
		Number[] c = ArrayUtils.append(a, 2.0);
		assertEquals(2, c.length);
		assertSame(Number[].class, c.getClass());
		assertSame(a[0], c[0]);
		assertEquals(2.0, c[1]);
	}

	@Test
	public void testAppendMixedTypes2() {
		Integer[] a = new Integer[] { 1 };
		Double[] b = new Double[] { 2.0 };
		Object[] c = ArrayUtils.append(a, b[0]);
		assertEquals(2, c.length);
		assertSame(Number[].class, c.getClass());
		assertSame(a[0], c[0]);
		assertSame(b[0], c[1]);
	}	
	
	@Test
	public void testAppendMixedTypes3() {
		Integer[] a = new Integer[] { 1 };
		Double[] b = new Double[] { 2.0 };
		Object[] c = ArrayUtils.append(Comparable.class, a, b[0]);
		assertEquals(2, c.length);
		assertSame(c.getClass(), Comparable[].class);
		assertSame(a[0], c[0]);
		assertSame(b[0], c[1]);
	}	
	
	@Test(expected = ClassCastException.class)
	public void testAppendMixedTypes4() {
		Integer[] a = new Integer[] { 1 };
		Double[] b = new Double[] { 2.0 };
		@SuppressWarnings("unused")
		Number[] c = (Number[])ArrayUtils.append(Object.class, a, b[0]);
	}
	
	@Test
	public void testAppendMixedTypes5() {
		// Java generic type inference does some fairly unpredictable stuff here
		Integer[] a = new Integer[] { 1 };
		String[] b = new String[] { "foo" };
		Object[] c = ArrayUtils.append(a, b[0]);
		// c is in fact not an Object array!
		// Java has determined that both Integer and String are instances of Comparable
		// Could be a Comparable array just as well...
		assert(Serializable[].class == c.getClass() || Comparable[].class == c.getClass());
		assertEquals(2, c.length);
		assertSame(a[0], c[0]);
		assertSame(b[0], c[1]);
	}

	@Test
	public void testAppendMixedTypes5b() {
		Object[] a = new Integer[] { 1 };
		Object[] b = new String[] { "foo" };
		// b[0] is passed as a VarArg of type Object
		Object[] c = ArrayUtils.append(a, b[0]);
		assertSame(Object[].class, c.getClass());
		assertEquals(2, c.length);
		assertSame(a[0], c[0]);
		assertSame(b[0], c[1]);
	}
	
	@Test
	public void testAppendMixedTypes6() {
		CharSequence[] a = new CharSequence[] { "bar" };
		String[] b = new String[] { "foo" };
		Object[] c = ArrayUtils.append(a, b[0]);
		assertSame(CharSequence[].class, c.getClass());
		assertEquals(2, c.length);
		assertSame(a[0], c[0]);
		assertSame(b[0], c[1]);
	}
	
	@Test
	public void testAppendMixedTypes6b() {
		Object[] a = new CharSequence[] { "bar" };
		Object[] b = new String[] { "foo" };
		// b[0] is passed as a VarArg of type Object
		Object[] c = ArrayUtils.append(a, b[0]);
		assertSame(Object[].class, c.getClass());
		assertEquals(2, c.length);
		assertSame(a[0], c[0]);
		assertSame(b[0], c[1]);
	}
	
	@Test
	public void testAppendMixedTypes7() {
		String[] a = new String[] { "bar" };
		CharSequence[] b = new CharSequence[] { "foo" };
		Object[] c = ArrayUtils.append(a, b[0]);
		assertSame(CharSequence[].class, c.getClass());
		assertEquals(2, c.length);
		assertSame(a[0], c[0]);
		assertSame(b[0], c[1]);
	}
	
	@Test
	public void testAppendMixedTypes7b() {
		Object[] a = new String[] { "bar" };
		Object[] b = new CharSequence[] { "foo" };
		Object[] c = ArrayUtils.append(a, b[0]);
		// Type inference presents b[0] as a VarArg of type Object
		assertSame(Object[].class, c.getClass());
		assertEquals(2, c.length);
		assertSame(a[0], c[0]);
		assertSame(b[0], c[1]);
	}
	
	@Test
	public void testAppendLengthGt() {
		Integer[] a = new Integer[] { 1, 2, 3 };
		Integer[] b = new Integer[] { 4, 5 };
		Integer[] c = ArrayUtils.append(8, a, b[0], b[1]);
		assertEquals(8, c.length);
		for (int i = 0; i < 5; i++)
			assertEquals(i+1, (int)c[i]);
		for (int i = 5; i < 8; i++)
			assertNull(c[i]);
		assertSame(a[0], c[0]);
		assertSame(b[0], c[3]);
	}

	@Test
	public void testAppendLengthLtB() {
		Integer[] a = new Integer[] { 1, 2, 3 };
		Integer[] b = new Integer[] { 4, 5 };
		Integer[] c = ArrayUtils.append(4, a, b[0], b[1]);
		assertEquals(4, c.length);
		for (int i = 0; i < 4; i++)
			assertEquals(i+1, (int)c[i]);
		assertSame(a[0], c[0]);
		assertSame(b[0], c[3]);
	}

	@Test
	public void testAppendLengthLtA() {
		Integer[] a = new Integer[] { 1, 2, 3 };
		Integer[] b = new Integer[] { 4, 5 };
		Integer[] c = ArrayUtils.append(2, a, b[0], b[1]);
		assertEquals(2, c.length);
		for (int i = 0; i < 2; i++)
			assertEquals(i+1, (int)c[i]);
		assertSame(a[0], c[0]);
	}
	
	@Test
	public void testAppendLengthZero() {
		Integer[] a = new Integer[] { 1, 2, 3 };
		Integer[] b = new Integer[] { 4, 5 };
		Integer[] c = ArrayUtils.append(0, a, b[0], b[1]);
		assertEquals(0, c.length);
	}
	
	@Test
	public void testAppendLengthZeroEmptyA() {
		Integer[] a = new Integer[] { };
		Integer[] b = new Integer[] { 4, 5 };
		Integer[] c = ArrayUtils.append(0, a, b[0], b[1]);
		assertEquals(0, c.length);
	}
	
	@Test
	public void testAppendLengthEmptyA() {
		Integer[] a = new Integer[] { };
		Integer[] b = new Integer[] { 4, 5 };
		Integer[] c = ArrayUtils.append(2, a, b[0], b[1]);
		assertEquals(2, c.length);
		assertArrayEquals(b,  c);
		assertNotSame(b, c);
	}
	
	@Test
	public void testAppendLengthEmptyB() {
		Integer[] a = new Integer[] { 1, 2, 3 };
		Integer[] c = ArrayUtils.append(3, a);
		assertEquals(3, c.length);
		assertArrayEquals(a,  c);
		assertNotSame(a, c);
	}
	
	@Test
	public void testGetCommonBase1() {
		Class<?> cl = ArrayUtils.getCommonBase(Integer.class, Double.class);
		assertSame(Number.class, cl);
	}
	
	@Test
	public void testGetCommonBase2() {
		Class<?> cl = ArrayUtils.getCommonBase(Integer[].class, Double[].class);
		assertSame(Number[].class, cl);
	}
	
	@Test
	public void testGetCommonBase3() {
		Class<?> cl = ArrayUtils.getCommonBase(Integer.class, Double[].class);
		assertSame(Object.class, cl);
	}
	
	@Test
	public void testGetCommonBase4() {
		Class<?> cl = ArrayUtils.getCommonBase(List.class, ArrayList.class);
		assertSame(List.class, cl);
	}
	
	@Test
	public void testGetCommonBase5() {
		Class<?> cl = ArrayUtils.getCommonBase(ArrayList.class, List.class);
		assertSame(List.class, cl);
	}
	
	@Test
	public void testGetCommonBase6() {
		Class<?> cl = ArrayUtils.getCommonBase(ArrayList.class, LinkedList.class);
		assertSame(AbstractList.class, cl);
	}
	
	@Test
	public void testGetCommonBase7() {
		Class<?> cl = ArrayUtils.getCommonBase(ArrayList.class, HashSet.class);
		assertSame(AbstractCollection.class, cl);
	}
	
	@Test
	public void testGetCommonBase8() {
		Iterable<Object> foo = new Iterable<Object>() {
			@Override
			public Iterator<Object> iterator() {
				return null;
			}
		};
		
		Class<?> cl2 = ArrayUtils.getCommonBase(Iterable.class, foo.getClass());
		Class<?> cl3 = ArrayUtils.getCommonBase(Iterable.class, ArrayList.class);
		Class<?> cl = ArrayUtils.getCommonBase(ArrayList.class, foo.getClass());
		
		assertSame(Iterable.class, cl2);
		assertSame(Iterable.class, cl3);
		assertSame(Object.class, cl);
	}
}
