package org.simantics.event.util;

import java.util.List;
import java.util.UUID;

import org.simantics.databoard.Bindings;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.PossibleTypedParent;
import org.simantics.db.exception.DatabaseException;
import org.simantics.event.ontology.EventResource;
import org.simantics.layer0.Layer0;

public class EventUtils {

	public static final int MAX_EVENTS = 1000;
	public static final int SLICE_SIZE = 64;
	public static final int KEEPS_SLICES = (MAX_EVENTS / SLICE_SIZE) + 2;
	public static final String MAX_SLICE_NAME = "" + (SLICE_SIZE-1);

	public static Resource createLog(WriteGraph graph) throws DatabaseException {
		Layer0 L0 = Layer0.getInstance(graph);
		EventResource EVENT = EventResource.getInstance(graph);
        Resource log = graph.newResource();
        graph.claim(log, L0.InstanceOf, null, EVENT.EventLog);
        graph.claimLiteral(log, L0.HasName, UUID.randomUUID().toString(), Bindings.STRING);
        graph.claimLiteral(log, EVENT.HasModificationCounter, 0, Bindings.INTEGER);
        return log;
	}

    /**
     * Bumps the modification counter value of the event log of the specified
     * events using {@link #bumpModificationCounter(WriteGraph, Resource)}.
     * <p>
     * The code assumes that all events are from the same log.
     * 
     * @param graph
     * @param forLogOfEvents
     * @throws DatabaseException
     */
    public static void bumpModificationCounter(WriteGraph graph, List<Resource> forLogOfEvents) throws DatabaseException {
        EventResource EVENT = EventResource.getInstance(graph);
        for (Resource event : forLogOfEvents) {
            Resource log = graph.syncRequest(new PossibleTypedParent(event, EVENT.EventLog));
            if (log != null) {
                bumpModificationCounter(graph, log);
                return;
            }
        }
    }

    /**
     * Bumps the modification counter of the specified event log by 1.
     * 
     * @param graph
     * @param eventLog
     * @return new modification counter value
     * @throws DatabaseException
     */
    public static int bumpModificationCounter(WriteGraph graph, Resource eventLog) throws DatabaseException {
        EventResource EVENT = EventResource.getInstance(graph);
        Resource counter = graph.getPossibleObject(eventLog, EVENT.HasModificationCounter);
        if (counter != null) {
            Integer c = graph.getValue(counter, Bindings.INTEGER);
            c = c == null ? 1 : c+1;
            graph.claimValue(counter, c, Bindings.INTEGER);
            return c;
        }
        return 0;
    }

}
