/*******************************************************************************
 * Copyright (c) 2014 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.event.util;

import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.util.Collection;
import java.util.Map;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.SubMonitor;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.event.view.contribution.EventLabelRule;
import org.simantics.event.view.contribution.ProjectEventsRule;
import org.simantics.utils.strings.EString;

/**
 * @author Tuukka Lehtonen
 */
public class EventExporter {

	/**
	 * SCL API.
	 * 
	 * @param outputFile
	 * @throws DatabaseException
	 * @throws IOException
	 */
	public static void exportCurrentEvents(String columnSeparator, String outputFile) throws DatabaseException, IOException {
		new EventExporter().exportCsv(null, new File(outputFile), columnSeparator);
	}

	/**
	 * SCL API.
	 * 
	 * @param outputFile
	 * @throws DatabaseException
	 * @throws IOException
	 */
	public static void exportCurrentEventsG(ReadGraph graph, String columnSeparator, String outputFile) throws DatabaseException, IOException {
		new EventExporter().exportCsv(graph, null, new File(outputFile), columnSeparator);
	}

	public EventExporter() {
	}

	/**
	 * @param monitor
	 *            the progress monitor to use for reporting progress to the
	 *            user. It is the caller's responsibility to call done() on the
	 *            given monitor. Accepts <code>null</code>, indicating that no
	 *            progress should be reported and that the operation cannot be
	 *            cancelled.</pre>
	 * @param file
	 * @param columnSeparator
	 * @throws DatabaseException
	 * @throws IOException
	 */
	public void exportCsv(IProgressMonitor monitor, File file, String columnSeparator) throws DatabaseException, IOException {
		try (PrintStream out = new PrintStream(file)) {
			Simantics.getSession().syncRequest(new ReadRequest() {
				@Override
				public void run(ReadGraph graph) throws DatabaseException {
					exportCsv(graph, monitor, out, columnSeparator);
				}
			});
		}
	}

	/**
	 * @param graph
	 * @param monitor
	 *            the progress monitor to use for reporting progress to the
	 *            user. It is the caller's responsibility to call done() on the
	 *            given monitor. Accepts <code>null</code>, indicating that no
	 *            progress should be reported and that the operation cannot be
	 *            cancelled.</pre>
	 * @param file
	 * @param columnSeparator
	 * @throws DatabaseException
	 * @throws IOException
	 */
	public void exportCsv(ReadGraph graph, IProgressMonitor monitor, File file, String columnSeparator) throws DatabaseException, IOException {
		try (PrintStream out = new PrintStream(file)) {
			exportCsv(graph, monitor, out, columnSeparator);
		}
	}

	/**
	 * @param graph
	 * @param monitor
	 *            the progress monitor to use for reporting progress to the
	 *            user. It is the caller's responsibility to call done() on the
	 *            given monitor. Accepts <code>null</code>, indicating that no
	 *            progress should be reported and that the operation cannot be
	 *            cancelled.</pre>
	 * @param out
	 * @param columnSeparator
	 * @throws DatabaseException
	 */
	public void exportCsv(ReadGraph graph, IProgressMonitor monitor, PrintStream out, String columnSeparator) throws DatabaseException {
		Collection<?> events = ProjectEventsRule.INSTANCE.getChildren(graph, Simantics.getProjectResource());
		if (events.isEmpty()) {
			out.println("No visible events recorded.");
			return;
		}

		final SubMonitor mon = SubMonitor.convert(monitor, "Export events", events.size());

		boolean first = true;
		for (Object obj : events) {
			if (mon.isCanceled())
				return;

			Resource event = (Resource) obj;
			Map<String, String> labels = EventLabelRule.INSTANCE.getLabel(graph, event, true);
			if (first) {
				out.println(EString.implode(labels.keySet(), columnSeparator));
				first = false;
			}
			out.println(EString.implode(labels.values(), columnSeparator));
			mon.worked(1);
		}
	}

}
