/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.charts.ui;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.Deque;

import org.eclipse.core.runtime.preferences.InstanceScope;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.preference.IPersistentPreferenceStore;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.ui.IExportWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.preferences.ScopedPreferenceStore;
import org.simantics.Simantics;
import org.simantics.db.management.ISessionContext;
import org.simantics.modeling.ui.Activator;
import org.simantics.modeling.ui.utils.NoProjectPage;
import org.simantics.project.IProject;
import org.simantics.project.ProjectKeys;
import org.simantics.utils.ui.ErrorLogger;
import org.simantics.utils.ui.ExceptionUtils;

/**
 * @author Antti Villberg
 */
public class CSVExportWizard extends Wizard implements IExportWizard {

    private static final int MAX_RECENT_EXPORT_PATHS = 10;

    Deque<String>            recentExportPaths;
    boolean                  overwrite;

    CSVExportPlan        plan;

    private boolean readPreferences() {
        IPreferenceStore store = new ScopedPreferenceStore(InstanceScope.INSTANCE, Activator.PLUGIN_ID);

        String recentPathsPref = store.getString(CSVExportPreferences.RECENT_LOCATIONS);
        recentExportPaths = CSVExportPreferences.decodePaths(recentPathsPref);
        overwrite = store.getBoolean(CSVExportPreferences.EXPORT_OVERWRITE);

        return true;
    }

    private void writePreferences() throws IOException {
        IPersistentPreferenceStore store = new ScopedPreferenceStore(InstanceScope.INSTANCE, Activator.PLUGIN_ID);

        store.putValue(CSVExportPreferences.RECENT_LOCATIONS, CSVExportPreferences.encodePaths(recentExportPaths));
        store.setValue(CSVExportPreferences.EXPORT_OVERWRITE, plan.overwrite);

        if (store.needsSaving())
            store.save();
    }

    public CSVExportWizard() {
        setWindowTitle("Export Subscription Data As CSV Text");
        setNeedsProgressMonitor(true);
    }

    @Override
    public void init(IWorkbench workbench, IStructuredSelection selection) {
        readPreferences();

        ISessionContext ctx = Simantics.getSessionContext();
        if (ctx == null)
            return;
        IProject project = ctx.getHint(ProjectKeys.KEY_PROJECT);
        if (project == null)
            return;

        plan = new CSVExportPlan(ctx, recentExportPaths);
        plan.project = project;
        plan.selection = selection;
        plan.overwrite = overwrite;
    }

    @Override
    public void addPages() {
        super.addPages();
        if (plan != null) {
            addPage(new CSVExportPage(plan));
        } else {
            addPage(new NoProjectPage("Export Subscription Data As CSV Text"));
        }
    }

    @Override
    public boolean performFinish() {
        try {
            recentExportPaths.addFirst(plan.exportLocation.getAbsolutePath());
            CSVExportPreferences.removeDuplicates(recentExportPaths);
            if (recentExportPaths.size() > MAX_RECENT_EXPORT_PATHS)
                recentExportPaths.pollLast();

            writePreferences();
        } catch (IOException e) {
            ErrorLogger.defaultLogError("Failed to write preferences", e);
        }

        final File output = plan.exportLocation;
        if (output.exists()) {
            if (!plan.overwrite) {
                boolean ok = MessageDialog.openConfirm(getShell(), "Overwrite", "A file by the name " + output.getAbsolutePath() + " already exists.\n\nDo you want to overwrite?");
                if (!ok) {
                    return false;
                }
            }
            if (!output.delete()) {
                MessageDialog.openError(getShell(), "Delete Problem", "Could not overwrite previously existing file " + output);
                return false;
            }
        }

        try {
            getContainer().run(true, true, new CSVExporter(plan));
        } catch (InvocationTargetException e) {
            Throwable t = e.getTargetException();
            WizardPage cp = (WizardPage) getContainer().getCurrentPage();
            if (t instanceof IOException) {
                ErrorLogger.defaultLogError("An I/O problem occurred while exporting subscription data. See exception for details.", t);
                cp.setErrorMessage("An I/O problem occurred while exporting subscription data.\n\nMessage: " + e.getMessage());
            } else {
                ErrorLogger.defaultLogError("Unexpected exception while exporting subscription data. See exception for details.", t);
                cp.setErrorMessage("Unexpected exception while exporting subscription data. See error log for details.\n\nMessage: " + e.getMessage());
            }
            return false;
        } catch (InterruptedException e) {
            ExceptionUtils.logAndShowError(e);
            return false;
        }

        return true;
    }

}
