package org.simantics.charts.query;

import java.util.Arrays;

import org.simantics.charts.ontology.ChartResource;
import org.simantics.charts.ui.ChartData;
import org.simantics.charts.ui.ChartData.ItemKey;
import org.simantics.databoard.Bindings;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.layer0.Layer0;
import org.simantics.trend.configuration.YAxisMode;

public class ChartDataWrite extends WriteRequest {

    ChartData data;
    Resource chart;

    public ChartDataWrite(Resource chart, ChartData data)
    {
        this.data = data;
        this.chart = chart;
    }

    @Override
    public void perform(WriteGraph g) throws DatabaseException {
        // 0. Initialization
        Layer0 L0 = Layer0.getInstance(g);
        ChartResource CHART = ChartResource.getInstance(g);

        // Label
        g.claimLiteral(chart, L0.HasName, data.name, Bindings.STRING);

        // Axis mode
        if ( data.axisMode == YAxisMode.SingleAxis ) {
            g.deny(chart, CHART.Chart_YAxisMode, CHART.YAxisMode_MultiAxis);
            g.claim(chart, CHART.Chart_YAxisMode, CHART.YAxisMode_SingleAxis);
        } else {
            g.deny(chart, CHART.Chart_YAxisMode, CHART.YAxisMode_SingleAxis);
            g.claim(chart, CHART.Chart_YAxisMode, CHART.YAxisMode_MultiAxis);
        }

        // Boolean flags
        g.claimLiteral(chart, CHART.Chart_ShowMilestones, data.showMilestones, Bindings.BOOLEAN);
        g.claimLiteral(chart, CHART.Chart_showGrid, data.showGrid, Bindings.BOOLEAN);
        g.claimLiteral(chart, CHART.Chart_trackExperimentTime, data.trackExperimentTime, Bindings.BOOLEAN);

        if ( data.timeIncrement!=null ) {
            g.claimLiteral(chart, CHART.Chart_TimeWindowIncrement, data.timeIncrement, Bindings.DOUBLE);
        } else {
            g.deny(chart, CHART.Chart_TimeWindowIncrement);
        }

        if ( data.timeLength==null ) {
            g.deny(chart, CHART.Chart_TimeWindowLength);
        } else {
            g.claimLiteral(chart, CHART.Chart_TimeWindowLength, data.timeLength, Bindings.DOUBLE);
        }
        if ( data.timeStart==null ) {
            g.deny(chart, CHART.Chart_TimeWindowStart);
        } else {
            g.claimLiteral(chart, CHART.Chart_TimeWindowStart, data.timeStart, Bindings.DOUBLE);
        }

        if ( data.allItems != null && data.hiddenItems != null ) {
            for (ItemKey item : data.allItems.keySet()) {
                boolean shouldBeHidden = data.hiddenItems.contains(item);
                Boolean isHidden = g.getRelatedValue(item.resource.getResource(), CHART.Chart_Item_hidden, Bindings.BOOLEAN);
                if (shouldBeHidden != isHidden)
                    g.claimLiteral(item.resource.getResource(), CHART.Chart_Item_hidden, shouldBeHidden, Bindings.BOOLEAN);
            }
        }

        if (data.backgroundColor1 == null) {
            g.denyValue(chart, CHART.Chart_backgroundColor);
        } else {
            float[] oldValue = g.getPossibleRelatedValue(chart, CHART.Chart_backgroundColor, Bindings.FLOAT_ARRAY);
            float[] newValue = data.backgroundGradient ? joinArrays(data.backgroundColor1, data.backgroundColor2) : data.backgroundColor1;
            if (newValue != null && (oldValue == null || !Arrays.equals(oldValue, newValue))) {
                g.claimLiteral(chart, CHART.Chart_backgroundColor, newValue, Bindings.FLOAT_ARRAY);
            }
        }

        if (data.gridColor == null) {
            g.denyValue(chart, CHART.Chart_gridColor);
        } else {
            float[] oldValue = g.getPossibleRelatedValue(chart, CHART.Chart_gridColor, Bindings.FLOAT_ARRAY);
            if (data.gridColor != null && (oldValue == null || !Arrays.equals(oldValue, data.gridColor))) {
                g.claimLiteral(chart, CHART.Chart_gridColor, data.gridColor, Bindings.FLOAT_ARRAY);
            }
        }
    }

    private static float[] joinArrays(float[] c1, float[] c2) {
        int l1 = c1 != null ? c1.length : 0;
        int l2 = c2 != null ? c2.length : 0;
        int l = l1 + l2;
        float[] result = new float[l];
        int i = 0;
        if (c1 != null) {
            System.arraycopy(c1, 0, result, 0, l1);
            i += l1;
        }
        if (c2 != null) {
            System.arraycopy(c2, 0, result, i, l2);
            i += l1;
        }
        return result;
    }

}
