package org.simantics.charts.internal;

import java.io.IOException;
import java.util.Optional;

import org.simantics.db.ReadGraph;
import org.simantics.db.RequestProcessor;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.Variables;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;

/**
 * @author Tuukka Lehtonen
 *
 */
public class JsonUtils {

    public static Optional<Variable> tryParseJsonPropertyVariable(RequestProcessor processor, String json) throws DatabaseException {
        return processor.syncRequest(new UniqueRead<Optional<Variable>>() {
            @Override
            public Optional<Variable> perform(ReadGraph graph) throws DatabaseException {
                return tryParseJsonPropertyVariable(graph, json);
            }
        });
    }

    public static Optional<Variable> tryParseJsonPropertyVariable(ReadGraph graph, String json) throws DatabaseException {
        try (JsonParser jp = new JsonFactory().createParser(json)) {
            return JsonUtils.readPossibleVariable(graph, jp);
        } catch (IOException e) {
            throw new DatabaseException(e);
        }
    }

    public static Optional<Variable> readPossibleVariable(ReadGraph graph, JsonParser jp) throws IOException, DatabaseException {
        Optional<String> uri = readPossibleVariableUri(jp);
        return uri.isPresent() ? Optional.ofNullable(Variables.getPossibleVariable(graph, uri.get())) : Optional.empty();
    }

    public static Optional<String> readPossibleVariableUri(JsonParser jp) throws IOException {
        // Sanity check: verify that we got "Json Object":
        if (jp.nextToken() != JsonToken.START_OBJECT)
            throw new IOException("Expected data to start with an Object");

        String uri = null;
        String type = null;
        String defaultPropertyUri = null;

        while (jp.nextToken() != JsonToken.END_OBJECT) {
            String fieldName = jp.getCurrentName();
            jp.nextToken();
            if (fieldName == null)
                continue;

            if (fieldName.equals("uri")) {
                uri = jp.getValueAsString();
            } else if (fieldName.equals("type")) {
                type = jp.getValueAsString();
            } else if (fieldName.equals("defaultPropertyUri")) {
                defaultPropertyUri = jp.getValueAsString();
            }
        }

        return Optional.ofNullable("Variable".equals(type) ?
                defaultPropertyUri != null ? defaultPropertyUri : uri
                        : null);
    }

}
