/*******************************************************************************
 * Copyright (c) 2007, 2012 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.swt;

import java.util.HashSet;

import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.TreeItem;
import org.simantics.browsing.ui.BuiltinKeys;
import org.simantics.browsing.ui.CheckedState;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.NodeQueryManager;
import org.simantics.browsing.ui.common.internal.GENodeQueryManager;
import org.simantics.browsing.ui.common.internal.IGraphExplorerContext;
import org.simantics.browsing.ui.content.PrunedChildrenResult;

/**
 * This class is responsible for updating the {@link TreeItem}s contained by the
 * SWT Tree in {@link GraphExplorerImpl}.
 * 
 * It will reschedule itself for as long as there are UI items to update in the
 * explorer. This class is not responsible for performing any throttling of
 * <code>UpdateRunner</code> executions.
 * 
 * @author Antti Villberg
 */
class UpdateRunner implements Runnable {

    final GraphExplorerImpl ge;
    //final IGraphExplorerContext geContext;

    UpdateRunner(GraphExplorerImpl ge, IGraphExplorerContext geContext) {
        this.ge = ge;
      //  this.geContext = geContext;
    }

    public void run() {
    	try {
    		doRun();
    	} catch (Throwable t) {
    		t.printStackTrace();
    	}
    	
    }

    public void doRun() {
    	
        if (ge.isDisposed())
            return;

        HashSet<TreeItem> items = new HashSet<TreeItem>();
        boolean doRoot;

        synchronized (ge.pendingItems) {

            doRoot = ge.pendingRoot;
            
            for (TreeItem item : ge.pendingItems) {
                //if(item.isDisposed()) continue;
                //if(item.getParentItem() == null) doRoot = true;
                //else items.add(item.getParentItem());
                if (item == null) {
                    doRoot = true;
                } else {
                    if(item.isDisposed()) continue;
                    items.add(item);
                }
            }

            ge.pendingItems = new HashSet<TreeItem>();
            ge.pendingRoot = false;

        }

        if (doRoot) {

            NodeQueryManager manager = new GENodeQueryManager(ge.explorerContext, null, null, TreeItemReference.create(null));

            PrunedChildrenResult children = manager.query(ge.rootContext, BuiltinKeys.PRUNED_CHILDREN);
            int maxChildren = ge.getMaxChildren(manager, ge.rootContext);
            int childCount = Math.min(children.getPrunedChildren().length, maxChildren);
            ge.tree.clearAll(true);
            ge.tree.setItemCount(childCount);

        } else {

            int itemIndex = 0;
            for (TreeItem item : items) {
                if (item.isDisposed())
                    continue;

                final NodeContext context = (NodeContext)item.getData();
                assert(context != null);

                NodeQueryManager manager = new GENodeQueryManager(ge.explorerContext, null, null, TreeItemReference.create(item));

                PrunedChildrenResult children = manager.query(context, BuiltinKeys.PRUNED_CHILDREN);

                int maxChildren = ge.getMaxChildren(manager, context);
                item.setItemCount(Math.min(children.getPrunedChildren().length, maxChildren));

                ge.setTextAndImage(item, manager, context, itemIndex);

                item.clearAll(true);

                boolean isExpanded = manager.query(context, BuiltinKeys.IS_EXPANDED);
                item.setExpanded(isExpanded);

                if ((((Control) ge.getControl()).getStyle() & SWT.CHECK) != 0) {
                    CheckedState checked = manager.query(context, BuiltinKeys.IS_CHECKED);
                    item.setChecked(CheckedState.CHECKED_STATES.contains(checked));
                    item.setGrayed(CheckedState.GRAYED == checked);
                }

                ++itemIndex;
            }
        }

        synchronized (ge.pendingItems) {
            if (!ge.scheduleUpdater())
                ge.updating = false;
        }
    }

}
