/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
/*
 * 12.6.2007
 */
package org.simantics.utils.threads.internal;

import java.lang.reflect.Array;

/**
 * ListenerList is an efficient and synchronized list of listeners.
 * The class is optimized for quick getListeners and slow add/remove.
 * 
 * @author Toni Kalajainen
 */
public class ListenerList<T> {
    
    /**
     * Array of listeners
     */
    private volatile T [] array;
    
    /** 
     * The class of T
     */
    private final Class<T> componentType;
           
    /**
     * Construct new Listener List
     * @param componentType the class of the listener type
     */
    public ListenerList(Class<T> componentType)
    {
        this.componentType = componentType;
        array = createArray(0);
    }
    
    public T[] getListeners()
    {
        return array;
    }
    
    public synchronized void add(T listener)
    {
        int oldLength = array.length;
        int newLength = oldLength + 1;
        T newArray[] = createArray(newLength);
        System.arraycopy(array, 0, newArray, 0, oldLength);
        newArray[oldLength] = listener;
        array = newArray;
    }
    
    /**
     * Removes the first occurance of listener.
     * If the listener is added multiple times, then it must be removed
     * as many times.
     * 
     * @param listener a listener
     * @return the listener that was removed from the list
     */
    public synchronized boolean remove(T listener)
    {
        int pos = getPos(listener);
        if (pos<0) return false;
        
        int oldLength = array.length;
        int newLength = oldLength -1;
        T newArray[] = createArray(newLength);
        
        // Copy beginning
        if (pos>0)
            System.arraycopy(array, 0, newArray, 0, pos);
        
        // Copy ending
        if (pos<newLength)
            System.arraycopy(array, pos+1, newArray, pos, newLength-pos);
        
        array = newArray;
        return true;
    }        
    
    private synchronized int getPos(T listener)
    {
        for (int i=0; i<array.length; i++)
            if (array[i] == listener)
                return i;
        return -1;
    }
    
    public int size()
    {
        return array.length;
    }
    
    public boolean isEmpty()
    {
        return array.length == 0;
    }
    
    public void clear()
    {
        array = createArray(0);
    }
    
    @SuppressWarnings("unchecked")
    private T[] createArray(int size)
    {
        return (T[]) Array.newInstance(componentType, size);
    }
    
}
