/*******************************************************************************
 * Copyright (c) 2007- VTT Technical Research Centre of Finland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
/*
 * Created on Jan 21, 2005
 * 
 * Copyright Toni Kalajainen
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.simantics.utils.bytes;

/**
 * Big Endian float <-> byte array conversions
 * Motorola order, Network order
 *
 * @author Toni Kalajainen
 */
public class BEFloat {
	
	/**
	 * Convert float to byte array
	 * @param l float value
	 * @return byte array
	 */
	public static byte[] toBytes(float value)
	{
		byte array[] = new byte[4];
		int l = Float.floatToIntBits(value);
		array[0] = (byte) (l & 0xff);
		array[1] = (byte) ((l >> 8) & 0xff);
		array[2] = (byte) ((l >> 16) & 0xff);
		array[3] = (byte) ((l >> 24) & 0xff);
		return array;
	}
	
	/**
	 * Write float value to byte array
	 * @param value the float value
	 * @param array the byte array
	 * @param offset the offset
	 */
	public static void write(float value, byte array[], int offset)
	{
		if (offset+4>array.length)
			throw new IndexOutOfBoundsException();		
		int l = Float.floatToIntBits(value);
		array[0 + offset] = (byte) (l & 0xff);
		array[1 + offset] = (byte) (l >> 8);
		array[2 + offset] = (byte) (l >> 16);
		array[3 + offset] = (byte) (l >> 24);
	}
	
	/**
	 * Write float value to byte array
	 * @param value the float value
	 * @param array the byte array
	 * @param offset the offset
	 */
	public static void write(float value, byte array[])
	{
		if (array.length<4)
			throw new IndexOutOfBoundsException();		
		int l = Float.floatToIntBits(value);
		array[0] = (byte) (l & 0xff);
		array[1] = (byte) (l >> 8);
		array[2] = (byte) (l >> 16);
		array[3] = (byte) (l >> 24);
	}	
	
	/**
	 * read float value from byte array
	 * @param array the array
	 * @param offset offset
	 * @return the value
	 */
	public static float toFloat(byte array[], int offset)
	{
		if (offset+4>array.length)
			throw new IndexOutOfBoundsException();		
		return 
			Float.intBitsToFloat(
			( ((int) array[0 + offset] & 0xFF) ) |
			( ((int) array[1 + offset] & 0xFF) << 8) |
			( ((int) array[2 + offset] & 0xFF) << 16) | 
			( ((int) array[3 + offset] & 0xFF) << 24)); 
	}
	
	/**
	 * read float value from byte array
	 * @param array the array
	 * @return the value
	 */
	public static float toFloat(byte array[])
	{
		if (4>array.length)
			throw new IndexOutOfBoundsException();		

		return 
		Float.intBitsToFloat(
		( ((int) array[0] & 0xFF) ) |
		( ((int) array[1] & 0xFF) << 8) |
		( ((int) array[2] & 0xFF) << 16) | 
		( ((int) array[3] & 0xFF) << 24)); 
	}

	/**
	 * Test cases
	 * @param args
	 */
	public static void main(String[] args) {
		System.out.println("min="+Float.MIN_VALUE+" max="+Float.MAX_VALUE);
		float value = -123.123123123f;
		byte array[] = toBytes(value);
		System.out.print(value);
		System.out.print(" = ");
		printByteArray(array);
		System.out.println();
		
		write(value, array, 0);
		System.out.print(value);
		System.out.print(" = ");
		printByteArray(array);
		System.out.println();
		
		write(value, array);
		System.out.print(value);
		System.out.print(" = ");
		printByteArray(array);
		System.out.println();
		
		value = toFloat(array, 0);
		printByteArray(array);
		System.out.print(" = ");
		System.out.print(value);
		System.out.println();
				
		value = toFloat(array);
		printByteArray(array);
		System.out.print(" = ");
		System.out.print(value);
		System.out.println();
		
	}
	
	public static void printByteArray(byte array[]) {
		for (int i=0; i<array.length; i++) {
			System.out.print(array[i] & 0xff);
			if (i<array.length-1) 
				System.out.print(",");
		}
	}
}
