/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.utils.datastructures.map;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.io.Serializable;
import java.util.Arrays;

/**
 * n-tuple (all fields are set) or n-tuple query (some fields possibly null)
 *
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public class Tuple implements Serializable, Externalizable {

	Object[] fields;
	private int hashCode;
	int associativity;
	
	Tuple() {}
	
	public Tuple(Object ... fields)
	{
		this.fields = fields;
		hashCode = Arrays.hashCode(fields);
		int mask = 1;
		for (Object o : fields) {
			if (o!=null) associativity |= mask;
			mask <<= 1;
		}
	}
	
	@Override
	public int hashCode() {
		return hashCode;
	}
	
	@Override
	public boolean equals(Object obj) {
        if (obj == null) return false;
        if (!(obj.getClass().equals(this.getClass()))) return false;        
		Tuple other = (Tuple) obj;
		return Arrays.deepEquals(fields, other.fields);
	}
	
    @Override
    public String toString() {
    	return Arrays.toString(fields);
    }
    
    public Object getField(int index)
    {
        return fields[index];
    }
    
    public Object tryGetField(int index)
    {
        return index >= 0 && index < fields.length ? fields[index] : null;
    }
    
    @SuppressWarnings("unchecked")
    public <T> T getTypedField(int index)
    {
    	return (T) fields[index];
    }
    
    @SuppressWarnings("unchecked")
    public <T> T tryGetTypedField(int index)
    {
        return index >= 0 && index < fields.length ? (T) fields[index] : null;
    }
    
    public Object[] getFields()
    {
    	return fields;
    }
    
    public int getLevel()
    {
    	return fields.length;
    }
    
    public int getAssociativity() 
    {
    	return associativity;
    }

	/**
	 * Tuple is full if all fields are set (not null) 
	 * @return <code>true</code> if all fields are set
	 */
	public boolean isFull()
	{
		return associativity == (1 << fields.length)-1;
	}
    
	@Override
	public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException {
		int count = in.readInt();
		fields = new Object[count];
		for (int i=0; i<count; i++)
			fields[i] = in.readObject();
		hashCode = Arrays.hashCode(fields);
	}

	@Override
	public void writeExternal(ObjectOutput out) throws IOException {
		out.writeInt(fields.length);
		for (Object o : fields)
			out.writeObject(o);
	}
		
}

