/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
/*
 *
 * @author Toni Kalajainen
 */
package org.simantics.utils.datastructures.hints;

import java.util.Map;

/**
 * Hint context interface
 *
 * @see Key
 * @see HintContext
 * @see HintStack
 * @see IHintObservable
 */
public interface IHintContext extends IHintObservable {

	/**
	 * Set a value for a hint. 
	 * @param key key
	 * @param value the value, <code>null</code> is not accepted
	 */
	void setHint(Key key, Object value);
	
	/**
	 * Set a set of hints
	 * @param hints
	 */
	void setHints(Map<Key, Object> hints);
	
	/**
	 * Remove a hint. In stack hint implementation lower priority hint will
	 * become effective. 

	 * @param <E> type of the object bound to the specified key
	 * @param key key to use for the hint
	 * @return the typed object currently bound to the specified key
	 */
	<E> E removeHint(Key key);

	/**
	 * Clears the context of all current hints without notifying any listeners.
	 */
	void clearWithoutNotification();

	/**
	 * Hint key
	 */
	public static abstract class Key {
		/**
		 * Test if value is accepted
		 * @param value the value
		 * @return true if value is accepted, false if not
		 */
		public abstract boolean isValueAccepted(Object value);		
	}
	
	/**
	 * Mouse id specific key. With this key there is unique value for each mouse id.
	 */
	public static abstract class MouseSpecificKey extends Key {
		private final int mouseId;
		public MouseSpecificKey(int mouseId)
		{
			this.mouseId = mouseId;
		}
		public int getMouseId()
		{
			return mouseId;
		}
		@Override
		public int hashCode() {
			return super.hashCode() ^ mouseId;
		}		
		@Override
		public boolean equals(Object obj) {
			if (!(obj instanceof MouseSpecificKey)) return false;
			if (((MouseSpecificKey)obj).mouseId != mouseId) return false;
			if (!obj.getClass().equals(getClass())) return false;
			return true;
		}
	}

	/**
	 * Key whose value is of a specific given class. 
	 */
	public static class KeyOf extends Key {
		final Class<?> clazz;
		final String keyName;
        public KeyOf(Class<?> clazz)
        {
            this.clazz = clazz;
            this.keyName = null;
        }
        public KeyOf(Class<?> clazz, String keyName)
        {
            this.clazz = clazz;
            this.keyName = keyName;
        }
		@Override
		public boolean isValueAccepted(Object value) {
			return clazz.isInstance(value);
		}
		@Override
		public String toString() {
		    if (keyName == null)
		        return "Key Of "+clazz.getName();
		    return keyName + "(" + clazz.getSimpleName() + ")";
		}
	}
	
	/**
	 * String based key
	 */
	public static abstract class StringKey extends Key {
		protected final String str;		
		protected int hash;
		public StringKey(String str)
		{
			this.str = str;
			hash = getClass().hashCode() ^str.hashCode();			
		}
		public String getString()
		{
			return str;
		}
		@Override
		public int hashCode() {
			return hash;
		}		
		@Override
		public boolean equals(Object obj) {
			if (!(obj instanceof StringKey)) return false;
			if (!((StringKey)obj).str.equals(str)) return false;
			if (obj.getClass()!=getClass()) return false;
			return true;
		}
	}	
	
	/**
	 * String based key whose value is of a specific class.
	 */
	public static class StringKeyOf extends StringKey {
		final Class<?> clazz;		
		public StringKeyOf(String str, Class<?> clazz)
		{
			super(str);
			this.clazz = clazz;
		}
		@Override
		public boolean isValueAccepted(Object value) {
			return clazz.isInstance(value);
		}
		@Override
		public String toString() {
			return "Key Of ("+clazz.getName()+", "+str+")";
		}		
	}
	

	public static class MouseSpecificKeyOf extends MouseSpecificKey {
		public final Class<?> clazz;
		public final int mouseId;
		final int hash;
		public MouseSpecificKeyOf(int mouseId, Class<?> clazz)
		{
			super(mouseId);
			this.clazz = clazz;
			this.mouseId = mouseId;
			hash = getClass().hashCode() ^ mouseId ^ 24392439;
		}
		@Override
		public boolean isValueAccepted(Object value) {
			return clazz.isInstance(value);
		}
		@Override
		public int hashCode() {
			return hash;
		}
		@Override
		public boolean equals(Object obj) {
			if (obj.getClass()!=this.getClass()) return false;
			MouseSpecificKeyOf other = (MouseSpecificKeyOf) obj;
			return other.mouseId == mouseId;
		}
	}
	
	
}
