/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.ui.workbench;

import org.eclipse.core.runtime.Assert;
import org.eclipse.ui.IViewPart;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.exception.InvalidResourceReferenceException;
import org.simantics.db.service.SerialisationSupport;
import org.simantics.utils.ui.workbench.WorkbenchUtils;

/**
 * Utilities for opening workbench views with a given input resource.
 * 
 * <p>
 * Since the only way of giving views input parameters is through the secondary
 * ID part of the whole view ID, we need to create random access ID's for any
 * resource references that need to be passed through the secondary ID. With
 * EditorParts this is not necessary right away because of IEditorInput which
 * can originally contain a ResourceReference.
 * </p>
 * 
 * @author Tuukka Lehtonen
 * 
 * @see ResourceInput
 */
public final class ResourceViewPartUtils {

    /**
     * Open a new ViewPart by its ID with a resource ID as a parameter to
     * that view. If there is an existing view with the same view ID and input
     * resource ID, it will be activated. Otherwise a new view will be opened.
     * 
     * @param viewId the workbench view ID of the editor to create
     * @param ls a valid Session for getting random access resource ID's
     *        and references.
     * @param inputResource a reference to resource to pass as a parameter to
     *        the specified view
     * @throws Exception any exception thrown while initializing the view
     */
    public static IViewPart activateViewForResource(String viewId, Session ls, Resource inputResource)
    throws Exception {
        return activateViewForResource(viewId, ls, inputResource, null);
    }

    /**
     * Open a new ViewPart by its ID with a resource ID as a parameter to
     * that view. If there is an existing view with the same view ID and input
     * resource ID, it will be activated. Otherwise a new view will be opened.
     * 
     * @param viewId the workbench view ID of the editor to create
     * @param ls a valid Session for getting random access resource ID's
     *        and references.
     * @param inputResource a reference to resource to pass as a parameter to
     *        the specified view
     * @param suffix <code>null</code> to ignore suffix
     * @throws Exception any exception thrown while initializing the view
     */
    public static IViewPart activateViewForResource(String viewId, Session ls, Resource inputResource, String suffix)
    throws Exception {
        Assert.isNotNull(viewId);
        Assert.isNotNull(ls);
        Assert.isNotNull(inputResource);

        ResourceInput input = newViewInput(ls, inputResource, suffix);
        final String param = viewId + ":" + input.marshall();
        return WorkbenchUtils.activateView(param);
    }

    /**
     * Open a new ViewPart by its ID with a resource ID as a parameter to that
     * view. If there is an existing view with the same view ID and input
     * resource ID, it will be activated. Otherwise a new view will be opened.
     * 
     * @param viewId the workbench view ID of the editor to create
     * @param perspectiveId the workbench perspective ID in which to open the
     *        editor or <code>null</code> to leave the perspective as is
     * @param ls a valid Session for getting random access resource ID's
     *        and references.
     * @param inputResource the resource reference to pass on to the view as a
     *        parameter
     * @param suffix <code>null</code> to ignore suffix
     * @throws Exception any exception thrown while initializing the view or
     *         showing the perspective
     */
    public static IViewPart activateViewForResourceInPerspective(String viewId, String perspectiveId, Session ls,
            Resource inputResource, String suffix) throws Exception {
        Assert.isNotNull(viewId);
        Assert.isNotNull(ls);
        Assert.isNotNull(inputResource);

        if (perspectiveId != null) {
            WorkbenchUtils.showPerspective(perspectiveId);
        }
        return activateViewForResource(viewId, ls, inputResource, suffix);
    }

    /**
     * Open a new ontology editor by its ID and the ID of the data model
     * resource to examine. A new view will be opened whether or not there is
     * already an existing view with the same ID and input resource ID.
     * 
     * @param viewId the workbench view ID of the editor to create
     * @param ls a valid Session for getting random access resource ID's
     *        and references.
     * @param inputResource the ID of the root core ID to pass on to the
     *        ontology editor
     * @param suffix <code>null</code> to ignore suffix
     * @throws Exception any exception thrown while initializing the view
     */
    public static IViewPart newViewForResource(String viewId, Session ls, Resource inputResource, String suffix)
    throws Exception {
        ResourceInput input = newViewInput(ls, inputResource, suffix);

        final String param = viewId + ":" + input.marshall();
        return WorkbenchUtils.activateView(param);

    }

    /**
     * Create a descriptor for the secondary ID of a workbench viewpart that
     * conveys the specified ResourceReference.
     * 
     * @param ls a valid Session for creating a random access ID for the
     *        input resource
     * @param r the resource reference to pass in the secondary ID of the view
     * @param suffix <code>null</code> to ignore suffix
     * @return an input descriptor for a secondary ID of a view for conveying
     *         the specified resource reference
     */
    public static ResourceInput newViewInput(Session ls, Resource r, String suffix) throws InvalidResourceReferenceException {
        String randomAccessId = getRandomAccessId(ls, r);
        return new ResourceInput(randomAccessId, suffix);
    }


    public static String getRandomAccessId(Session s, Resource r)
    throws InvalidResourceReferenceException {
        SerialisationSupport support = s.getService(SerialisationSupport.class);
        return support.getResourceSerializer().createRandomAccessId(r);
    }

}
