/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.core.runtime.preferences.DefaultScope;
import org.eclipse.core.runtime.preferences.InstanceScope;
import org.osgi.service.prefs.Preferences;
import org.simantics.db.ReadGraph;
import org.simantics.db.Session;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.service.LifecycleSupport;
import org.simantics.db.service.VirtualGraphSupport;
import org.simantics.internal.Activator;

/**
 * @author Tuukka Lehtonen <tuukka.lehtonen@semantum.fi>
 */
public class AutosaveVirtualGraphs extends Job {

	private static final boolean         TRACE        = false;

	private boolean                      enabled      = true;

	private Preferences                  defaultPrefs = DefaultScope.INSTANCE.getNode(AutosavePreferences.P_NODE);
	private Preferences                  prefs        = InstanceScope.INSTANCE.getNode(AutosavePreferences.P_NODE);

	private static AutosaveVirtualGraphs INSTANCE;

	private boolean getBooleanPref(String preference, boolean def) {
		return prefs.getBoolean(preference, defaultPrefs.getBoolean(preference, def));
	}

	private int getIntPref(String preference, int def) {
		return prefs.getInt(preference, defaultPrefs.getInt(preference, def));
	}

	private AutosaveVirtualGraphs() {
		super("Autosave Virtual Graphs");
		setSystem(true);
		setPriority(Job.LONG);
	}

	public static synchronized AutosaveVirtualGraphs getInstance() {
		if(INSTANCE == null) {
			INSTANCE = new AutosaveVirtualGraphs();
		}
		return INSTANCE;
	}

	/**
	 * @param enabled
	 * @return
	 */
	public boolean isEnabled() {
		return enabled;
	}

	/**
	 * @param enabled new enabled state
	 * @return previous enabled state
	 */
	public boolean setEnabled(boolean enabled) {
		if (enabled == this.enabled)
			return enabled;
		this.enabled = enabled;
		if (enabled)
			scheduleAfterInterval();
		else
			cancel();
		return !enabled;
	}

	public AutosaveVirtualGraphs scheduleAfterInterval() {
		// Check global disable first.
		String enabled = System.getProperty(AutosavePreferences.SYSTEM_PROPERTY_AUTOSAVE);
		if (enabled != null && enabled.equalsIgnoreCase(Boolean.FALSE.toString())) {
			return this;
		}

		// Then check preference.
		if (!getBooleanPref(AutosavePreferences.P_VG_AUTOSAVE_ENABLED, AutosavePreferences.DEFAULT_VG_AUTOSAVE_ENABLED))
			return this;

		wakeUp();
		int interval = getIntPref(AutosavePreferences.P_VG_AUTOSAVE_INTERVAL, AutosavePreferences.DEFAULT_VG_AUTOSAVE_INTERVAL);
		schedule(interval*1000L);
		return this;
	}

	@Override
	public boolean shouldSchedule() {
		return enabled;
	}

	@Override
	public boolean shouldRun() {
		return enabled;
	}

	protected IStatus runHeadless(IProgressMonitor monitor) {
		try {
			Session session = Simantics.peekSession();
			if (session == null)
				return Status.CANCEL_STATUS;
			LifecycleSupport lfs = session.peekService(LifecycleSupport.class);
			if (lfs == null || lfs.isClosed() || lfs.isClosing())
				return Status.CANCEL_STATUS;

			// Save
			monitor.beginTask("Autosaving virtual graphs...", IProgressMonitor.UNKNOWN);
			if (TRACE)
				System.out.println("Autosaving virtual graphs...");
			long startTime = System.nanoTime();

			session.syncRequest(new ReadRequest() {
				@Override
				public void run(ReadGraph graph) throws DatabaseException {
					doSave(graph);
				}
			});

			if (TRACE) {
				long endTime = System.nanoTime();
				System.out.println("Autosave of virtual graphs completed in " + (endTime-startTime)*1e-6 + " ms");
			}

			return Status.OK_STATUS;
		} catch (DatabaseException e) {
			return new Status(Status.ERROR, Activator.PLUGIN_ID, "Autosaving virtual graphs failed", e);
		} finally {
		}
	}

	@Override
	protected IStatus run(IProgressMonitor monitor) {
		if (!getBooleanPref(AutosavePreferences.P_VG_AUTOSAVE_ENABLED, AutosavePreferences.DEFAULT_VG_AUTOSAVE_ENABLED))
			return Status.OK_STATUS;
		int interval = getIntPref(AutosavePreferences.P_VG_AUTOSAVE_INTERVAL, AutosavePreferences.DEFAULT_VG_AUTOSAVE_INTERVAL);
		try {
			// Never run while a heavy database job is in progress.
			if (DatabaseJob.inProgress()) {
				// Schedule again in at most 10 seconds instead of
				// waiting for the whole autosave period again.
				interval = Math.min(10, interval);
				return Status.OK_STATUS;
			}

//			if(PlatformUI.isWorkbenchRunning()) {
//				return runInWorkbench(monitor);
//			} else {
				return runHeadless(monitor);
//			}
		} finally {
			schedule(interval*1000L);
		}
	}

	public static void doSave(ReadGraph graph) throws DatabaseException {
		VirtualGraphSupport vgs = graph.getService(VirtualGraphSupport.class);
		vgs.saveAll();
	}

	public static void saveVirtualGraphsPeriodically() {
		getInstance().scheduleAfterInterval();
	}

}
