/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.selectionview;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.dynamichelpers.ExtensionTracker;
import org.eclipse.core.runtime.dynamichelpers.IExtensionChangeHandler;
import org.eclipse.core.runtime.dynamichelpers.IExtensionTracker;
import org.eclipse.core.runtime.dynamichelpers.IFilter;
import org.simantics.browsing.ui.common.Activator;
import org.simantics.browsing.ui.common.ErrorLogger;
import org.simantics.utils.strings.StringUtils;

/**
 * @author Tuukka Lehtonen
 */
public class SelectionProcessorExtensionManager implements IExtensionChangeHandler {

    private final static String   NAMESPACE  = Activator.PLUGIN_ID;

    private final static String   EP_NAME    = "selectionProcessor";

    private ExtensionTracker      tracker;

    SelectionProcessorExtension[] extensions = new SelectionProcessorExtension[0];

    SelectionProcessorExtensionManager() {
        tracker = new ExtensionTracker();

        // Cache defined actions
        IExtensionPoint expt = Platform.getExtensionRegistry().getExtensionPoint(NAMESPACE, EP_NAME);
        loadExtensions(expt.getConfigurationElements());

        // Start tracking for new and removed extensions
        IFilter filter = ExtensionTracker.createExtensionPointFilter(expt);
        tracker.registerHandler(this, filter);
    }

    void close() {
        tracker.close();
        tracker = null;
        extensions = new SelectionProcessorExtension[0];
    }

    public SelectionProcessorExtension[] getExtensions() {
        return extensions;
    }

    private void loadExtensions(IConfigurationElement[] elements) {

        Set<SelectionProcessorExtension> newExtensions = new HashSet<SelectionProcessorExtension>(Arrays.asList(extensions));

        for (IConfigurationElement el : elements) {
            try {

                String id = StringUtils.safeString(el.getAttribute("id"));
                SelectionProcessor<?, ?> factory = (SelectionProcessor<?, ?>) el.createExecutableExtension("class");
                SelectionProcessorExtension ext = new SelectionProcessorExtensionImpl(id, factory);

                // Start tracking the new extension object, its removal will be notified of
                // with removeExtension(extension, Object[]).
                tracker.registerObject(el.getDeclaringExtension(), ext, IExtensionTracker.REF_STRONG);

                newExtensions.add(ext);
            } catch (CoreException e) {
                ErrorLogger.defaultLogError("Failed to initialize " + EP_NAME + " extension \"" + el.getName()
                        + "\" with name \"" + el.getAttribute("name") + "\": "
                        + e.getMessage(), e);
            }
        }

        // Atomic assignment
        this.extensions = newExtensions.toArray(new SelectionProcessorExtension[newExtensions.size()]);
    }

    @Override
    public void addExtension(IExtensionTracker tracker, IExtension extension) {
        loadExtensions(extension.getConfigurationElements());
    }

    @Override
    public void removeExtension(IExtension extension, Object[] objects) {
        Set<SelectionProcessorExtension> newExtensions = new HashSet<SelectionProcessorExtension>(Arrays.asList(extensions));

        for (Object o : objects) {
            tracker.unregisterObject(extension, o);
            newExtensions.remove(o);
        }

        // Atomic assignment
        this.extensions = newExtensions.toArray(new SelectionProcessorExtension[newExtensions.size()]);
    }

    private static SelectionProcessorExtensionManager INSTANCE;

    public static synchronized SelectionProcessorExtensionManager getInstance() {
        if (INSTANCE == null)
            INSTANCE = new SelectionProcessorExtensionManager();
        return INSTANCE;
    }


    public static synchronized void dispose() {
        if (INSTANCE != null) {
            INSTANCE.close();
            INSTANCE = null;
        }
    }

    public static SelectionProcessor<?, ?> getProcessor(String id) {
        for (SelectionProcessorExtension ext : getInstance().extensions) {
            if (ext.getProcessor().getClass().getCanonicalName().equals(id))
                return ext.getProcessor();
        }
        return null;
    }

}
