/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.selectionview;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.dynamichelpers.ExtensionTracker;
import org.eclipse.core.runtime.dynamichelpers.IExtensionChangeHandler;
import org.eclipse.core.runtime.dynamichelpers.IExtensionTracker;
import org.eclipse.core.runtime.dynamichelpers.IFilter;
import org.simantics.browsing.ui.common.Activator;
import org.simantics.scl.reflection.OntologyVersions;
import org.simantics.utils.strings.StringUtils;

/**
 * @author Tuukka Lehtonen
 */
public class SelectionProcessorBindingExtensionManager implements IExtensionChangeHandler {

    private final static String                  NAMESPACE  = Activator.PLUGIN_ID;

    private final static String                  EP_NAME    = "selectionProcessorBinding";

    private ExtensionTracker                     tracker;

    private SelectionProcessorBindingExtension[] extensions = new SelectionProcessorBindingExtension[0];

    SelectionProcessorBindingExtensionManager() {
        tracker = new ExtensionTracker();

        // Cache defined actions
        IExtensionPoint expt = Platform.getExtensionRegistry().getExtensionPoint(NAMESPACE, EP_NAME);
        loadExtensions(expt.getConfigurationElements());

        // Start tracking for new and removed extensions
        IFilter filter = ExtensionTracker.createExtensionPointFilter(expt);
        tracker.registerHandler(this, filter);
    }

    void close() {
        tracker.close();
        tracker = null;
        extensions = new SelectionProcessorBindingExtension[0];
    }

    public SelectionProcessorBindingExtension[] getExtensions() {
        return extensions;
    }

    SelectionProcessorReferenceBinding createReferenceBinding(String browseContext, String factoryId) {
        return new SelectionProcessorReferenceBinding(browseContext, factoryId);
    }


    private void loadExtensions(IConfigurationElement[] elements) {

        Set<SelectionProcessorBindingExtension> newExtensions = new HashSet<SelectionProcessorBindingExtension>(Arrays.asList(extensions));

        for (IConfigurationElement el : elements) {

            String browseContext = OntologyVersions.getInstance().currentVersion(StringUtils.safeString(el.getAttribute("browseContext")));

            for (IConfigurationElement child : el.getChildren("reference")) {

                String factoryId = StringUtils.safeString(child.getAttribute("id"));
                SelectionProcessorBindingExtension ext = createReferenceBinding(browseContext, factoryId);

                // Start tracking the new extension object, its removal will be notified of
                // with removeExtension(extension, Object[]).
                tracker.registerObject(el.getDeclaringExtension(), ext, IExtensionTracker.REF_STRONG);

                newExtensions.add(ext);

            }

            for (IConfigurationElement child : el.getChildren("implementation")) {

                try {

                    SelectionProcessor<?, ?> processor = (SelectionProcessor<?, ?>) child.createExecutableExtension("class");
                    SelectionProcessorBindingExtension ext = new SelectionProcessorImplementationBinding(browseContext, processor);

                    // Start tracking the new extension object, its removal will be notified of
                    // with removeExtension(extension, Object[]).
                    tracker.registerObject(el.getDeclaringExtension(), ext, IExtensionTracker.REF_STRONG);

                    newExtensions.add(ext);

                } catch (CoreException e) {

                    System.out.println(" == Could not load ViewpointContributionFactory '" + child.getAttribute("class") + "' due to the following error: " + e.getMessage()  );

                }

            }

        }

        // Atomic assignment
        this.extensions = newExtensions.toArray(new SelectionProcessorBindingExtension[newExtensions.size()]);
    }

    @Override
    public void addExtension(IExtensionTracker tracker, IExtension extension) {
        loadExtensions(extension.getConfigurationElements());
    }

    @Override
    public void removeExtension(IExtension extension, Object[] objects) {
        Set<SelectionProcessorBindingExtension> newExtensions = new HashSet<SelectionProcessorBindingExtension>(Arrays.asList(extensions));

        for (Object o : objects) {
            tracker.unregisterObject(extension, o);
            newExtensions.remove(o);
        }

        // Atomic assignment
        this.extensions = newExtensions.toArray(new SelectionProcessorBindingExtension[newExtensions.size()]);
    }

    public Set<SelectionProcessorBinding> getBoundContributions(Set<String> browseContexts) {
        HashSet<SelectionProcessorBinding> result = new HashSet<SelectionProcessorBinding>();

        for (SelectionProcessorBindingExtension binding : getExtensions()) {
            if (browseContexts.contains(binding.getBrowseContext())) {
                SelectionProcessor<?, ?> processor = binding.getProcessor();
                if (processor != null) {
//                    System.out.println("----------- Plugin contribution " + binding.getFactory());
                    result.add(new SelectionProcessorBindingImpl(processor));
                } else {
//                    System.out.println("FAILED: ----------- No factory for " + binding);
                }
            }
        }

        return result;
    }

    private static SelectionProcessorBindingExtensionManager INSTANCE;

    public static synchronized SelectionProcessorBindingExtensionManager getInstance() {
        if (INSTANCE == null)
            INSTANCE = new SelectionProcessorBindingExtensionManager();
        return INSTANCE;
    }

    public static synchronized void dispose() {
        if (INSTANCE != null) {
            INSTANCE.close();
            INSTANCE = null;
        }
    }

}
