/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.selectionview;

import java.util.function.Consumer;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.ui.IPartListener2;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPartReference;
import org.eclipse.ui.IWorkbenchPartSite;
import org.eclipse.ui.part.IPageSite;
import org.simantics.Simantics;
import org.simantics.browsing.ui.common.IPageBookViewPagePartInit;
import org.simantics.browsing.ui.common.views.IFilterAreaProvider;
import org.simantics.browsing.ui.swt.PartNameListener;
import org.simantics.browsing.ui.swt.PropertyPageUtil;
import org.simantics.db.ReadGraph;
import org.simantics.db.Session;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.management.ISessionContext;
import org.simantics.db.management.ISessionContextChangedListener;
import org.simantics.db.management.ISessionContextProvider;
import org.simantics.db.management.SessionContextChangedEvent;
import org.simantics.db.request.Read;
import org.simantics.ui.workbench.IPropertyPage;

/**
 * <p>
 * Subclasses may extend or reimplement the following methods as required:
 * <ul>
 *   <li><code>createPageControls</code> - to create the page's controls</li>
 *   <li><code>fillToolBar</code> - to add actions to the page's toolbar</li>
 *   <li><code>fillDropDownMenu</code> - to add actions to the page's drop-down menu</li>
 *   <li><code>getControl</code> - to retrieve the page's control</li>
 *   <li><code>setFocus</code> - implement to accept focus</li>
 *   <li><code>sourceSelectionChanged</code> - puts the incoming ISelection into use in this page</li>
 *   <li><code>sourcePartClosed</code> - cleans up the page controls after a current selection source part has been closed</li>
 *   <li><code>dispose</code> - extend to provide additional cleanup</li>
 *   <li><code>setActionBars</code> - reimplement to make contributions</li>
 *   <li><code>makeContributions</code> - this method exists to support previous versions</li>
 *   <li><code>setActionBars</code> - this method exists to support previous versions</li>
 *   <li><code>init</code> - extend to provide additional setup for the view before any controls are created</li>
 *   <li><code>sessionContextChanged</code> - reimplement to take actions when the source database session changes</li>
 * </ul>
 * </p>
 * 
 * @author Tuukka Lehtonen
 */
public class PropertyPage extends AbstractPropertyPage implements IPropertyPage, IAdaptable, IPartListener2, IPageBookViewPagePartInit {

    protected static final int MAX_SELECTION_LENGTH_TO_SHOW = PropertyPageUtil.MAX_SELECTION_LENGTH_TO_SHOW;

    protected ISessionContext  sessionContext;

    protected IPropertyTab     tab;

    protected ISelection       latestSelection = StructuredSelection.EMPTY;

    protected IWorkbenchPart   propertyPageView;
    
    protected boolean visible = true;

    /**
     * @param site the workbench part site that contains this page or
     *        <code>null</code> if there is no site, i.e. the page is within a
     *        dialog or a plain shell.
     */
    public PropertyPage(IWorkbenchPartSite site) {
    	super(site);
    }

    @Override
    public void init(IPageSite pageSite) {
    	
    	super.init(pageSite);
    	
    	if(getSite() != null)
    		getSite().getPage().addPartListener(this);
    	
    }
    
    @Override
    public void initPart(IWorkbenchPart part) {
    	propertyPageView = part;
    }
    
    @Override
    public void dispose() {

    	if(getSite() != null)
    		getSite().getPage().removePartListener(this);
    	
        // Stop listening for title changes.
        if (currentPartNameListener != null) {
            currentPartNameListener.dispose();
            currentPartNameListener = null;
        }

        ISessionContextProvider contextProvider = getSessionContextProvider();
        if (contextProvider != null)
            contextProvider.removeContextChangedListener(contextChangeListener);

        //System.out.println("dispose " + this);
        // Must invoke dispose before nullifying table since super will dispose the widget.
        super.dispose();
        tab = null;
        sessionContext = null;
    }

    protected ISessionContextProvider getSessionContextProvider() {
        return Simantics.getSessionContextProvider();
    }

    protected ISessionContext getSessionContext() {
        return sessionContext;
    }

    protected Session getSession() {
        if (sessionContext == null)
            throw new IllegalStateException("null session context");
        return sessionContext.getSession();
    }

    @Override
    public final void createControl(Composite parent) {
        createPageControls(parent);

        // Attach to current session context provider to keep the UI intact even
        // when the current UI session changes.
        ISessionContextProvider contextProvider = getSessionContextProvider();
        contextProvider.addContextChangedListener(contextChangeListener);
        setSessionContext(contextProvider.getSessionContext());
    }

    /**
     * Override to customize the UI component created on this page.
     * 
     * @param parent
     */
    protected void createPageControls(Composite parent) {
        PropertyTable pt = new PropertyTable(sourceSite, parent, SWT.NONE);
        tab = pt;
        tab.createControl(pt, getSessionContext());
        tab.getControl().addListener(SWT.Dispose, new Listener() {
            @Override
            public void handleEvent(Event event) {
                PropertyPage.this.dispose();
            }
        });
        ISelectionProvider pv = pt.getSelectionProvider();
        getSite().setSelectionProvider(pv);

        fillToolBar(getSite().getActionBars().getToolBarManager());
        fillDropDownMenu(getSite().getActionBars().getMenuManager());
    }

    protected void fillToolBar(IToolBarManager manager) {
    }

    protected void fillDropDownMenu(IMenuManager manager) {
    }

    /**
     * @param newContext
     */
    protected final void setSessionContext(ISessionContext newContext) {
//        ISessionContext oldContext = this.sessionContext;
        this.sessionContext = newContext;
//        System.out.println("PropertyPage.setSessionContext: " + oldContext + " -> " + newContext);
    }

    protected ISessionContextChangedListener contextChangeListener = new ISessionContextChangedListener() {
        @Override
        public void sessionContextChanged(SessionContextChangedEvent event) {
            setSessionContext(event.getNewValue());
        }
    };

    @Override
    public Control getControl() {
        return (tab != null) ? tab.getControl() : null;
    }

    @Override
    public ISelection getSelection() {
        if (tab != null && !tab.isDisposed()) {
            ISelectionProvider provider = tab.getSelectionProvider();
            if (provider != null)
                return provider.getSelection();
        }
        return null;
    }

    /**
     * Sets focus to a part in the page.
     * @see org.eclipse.ui.part.Page#setFocus()
     */
    @Override
    public void setFocus() {
        if (tab != null && !tab.isDisposed()) {
            tab.requestFocus();
        }
    }

    @Override
    protected void sourcePartClosed(IWorkbenchPart part) {
        if (tab != null && !tab.isDisposed()) {
            tab.setInput(getSessionContext(), StructuredSelection.EMPTY, false);
        }
    }

    @Override
    protected void sourceSelectionChanged(ISelection selection) {
        if (tab != null && !tab.isDisposed()) {
            latestSelection = selection;
            if(visible)
                tab.setInput(getSessionContext(), selection, false);
        }
    }

    protected void refresh() {
        if (tab != null && !tab.isDisposed()) {
            tab.setInput(getSessionContext(), latestSelection, true);
        }
    }

    protected PartNameListener currentPartNameListener = null;

    @Override
    public void updatePartName(final ISelection forSelection, Consumer<String> updateCallback) {
    	
    	if(!visible) {
    		updateCallback.accept("Selection");
    		return;
    	}
    	
        PartNameListener oldListener = currentPartNameListener;
        PartNameListener newListener = new PartNameListener(updateCallback);
        currentPartNameListener = newListener;
        if (oldListener != null)
            oldListener.dispose();

        if (sessionContext != null) {
            sessionContext.getSession().asyncRequest(new Read<String>() {
                @Override
                public String perform(ReadGraph graph) throws DatabaseException {
                    return computeTitle(graph, forSelection);
                }
            }, newListener);
        }
    }

    protected String computeTitle(ReadGraph graph, ISelection forSelection) throws DatabaseException {
        return PropertyPageUtil.computeTitle(graph, forSelection);
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T> T getAdapter(Class<T> adapter) {
        if (adapter == IFilterAreaProvider.class)
            return (T) tab;
        return null;
    }

	@Override
	public void partActivated(IWorkbenchPartReference partRef) {
	}

	@Override
	public void partBroughtToTop(IWorkbenchPartReference partRef) {
	}

	@Override
	public void partClosed(IWorkbenchPartReference partRef) {
	}

	@Override
	public void partDeactivated(IWorkbenchPartReference partRef) {
	}

	@Override
	public void partOpened(IWorkbenchPartReference partRef) {
	}

	@Override
	public void partInputChanged(IWorkbenchPartReference partRef) {
	}
	
    @Override
    public void partHidden(IWorkbenchPartReference partRef) {
        IWorkbenchPart part = partRef.getPart(false);
        if (propertyPageView.equals(part)) {
        	visible = false;
        }
    }

    @Override
    public void partVisible(IWorkbenchPartReference partRef) {
        IWorkbenchPart part = partRef.getPart(false);
        if (propertyPageView.equals(part)) {
        	visible = true;
        	sourceSelectionChanged(latestSelection);
        	//System.err.println("[" + this + "]selection view was shown");
        }
    }
	
}
