/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.scenegraph.g2d.nodes;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Paint;
import java.awt.Stroke;
import java.awt.geom.Line2D;
import java.awt.geom.Rectangle2D;

import org.simantics.scenegraph.g2d.G2DNode;
import org.simantics.scenegraph.utils.GeometryUtils;

/**
 * @author Tuukka Lehtonen
 */
public class PageBorderNode extends G2DNode {

    private static final long  serialVersionUID = -5969105979396122487L;

    private static final Paint MARGIN_COLOR     = new Color(192, 128, 128);

    protected Rectangle2D      border           = null;
    protected Rectangle2D      margin           = null;
    protected Boolean          enabled          = null;

    @SyncField( { "border", "margin", "enabled" })
    public void init(Rectangle2D border, Rectangle2D margin, Boolean enabled) {
        //System.out.println("PageBorderNode.init(" + border + ", " + margin + ", " + enabled);
        this.border = border;
        this.margin = margin;
        this.enabled = enabled;
    }

    @SyncField("enabled")
    public void setEnabled(Boolean enabled) {
        this.enabled = enabled;
    }

    @Override
    public Rectangle2D getBounds() {
        if (border != null)
            return border;
        if (margin != null)
            return margin;
        return null;
    }

    @Override
    public void render(Graphics2D g) {
        if (Boolean.FALSE.equals(enabled))
            return;

        boolean borderPaintable = isPaintable(border);
        boolean marginPaintable = isPaintable(margin);
        if (!borderPaintable && !marginPaintable)
            return;

        double scaleRecip = 1 / GeometryUtils.getScale(g.getTransform());
        Stroke s = new BasicStroke((float) (1 * scaleRecip));

        boolean borderPainted = false;
        if (borderPaintable) {
            Stroke ss = new BasicStroke((float) (2 * scaleRecip));
            paintRectangle(g, scaleRecip, border, s, Color.BLACK, ss, Color.GRAY);
            borderPainted = true;
        }

        if (marginPaintable && (!borderPainted || !margin.equals(border))) {
            paintRectangle(g, scaleRecip, margin, s, MARGIN_COLOR, null, null);
        }
    }

    boolean isPaintable(Rectangle2D r) {
        return r != null && !r.isEmpty() && r.getWidth() != Double.POSITIVE_INFINITY
        && r.getHeight() != Double.POSITIVE_INFINITY;
    }

    void paintRectangle(Graphics2D g, double scaleRecip, Rectangle2D r, Stroke stroke, Paint strokePaint,
            Stroke shadowStroke, Paint shadowPaint) {
        g.setPaint(strokePaint);
        g.setStroke(stroke);
        g.draw(r);

        if (shadowStroke != null) {
            g.setPaint(shadowPaint);
            g.setStroke(shadowStroke);
            double offset = 2 * scaleRecip;
            double w = r.getWidth();
            double h = r.getHeight();
            double x = r.getMaxX() + offset;
            double x1 = r.getMinX() + offset * 2;
            double y1 = r.getMinY() + offset * 2;
            g.draw(new Line2D.Double(x, y1, x, y1 + h - offset));
            double y = r.getMaxY() + offset;
            g.draw(new Line2D.Double(x1, y, x1 + w - offset, y));
        }
    }

    @Override
    public Rectangle2D getBoundsInLocal() {
        return border;
    }

    @Override
    public String toString() {
        return super.toString() + " [enabled=" + enabled + ", border=" + border + ", margin=" + margin + "]";
    }

}
