/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.project.features.registry;

import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

/**
 * A reference to a P2 feature group installable unit (IU).
 * 
 * <p>
 * Contains an ID and an optional version specifier. If a version is not
 * specified, it is <code>null</code>. The version can be either a single
 * version or a version range.
 * 
 * <p>
 * Serializes to the format <code>ID/version</code>
 * 
 * @author Tuukka Lehtonen
 */
public class GroupReference implements Comparable<GroupReference> {

    public static final GroupReference OMNIPRESENT = new GroupReference("omnipresent");

    private final String id;
    private final String version;

    /**
     * @return
     */
    public GroupReference withoutVersion() {
        return version == null ? this : new GroupReference(id);
    }

    /**
     * @param refs
     * @return
     */
    public static Collection<GroupReference> stripVersions(Collection<GroupReference> refs) {
        Set<GroupReference> result = new HashSet<GroupReference>();
        for (GroupReference ref : refs)
            result.add( ref.withoutVersion() );
        return result;
    }

    /**
     * @param ref
     * @return
     */
    public static GroupReference parse(String ref) {
        // TODO: better validation
        int slash = ref.indexOf('/');
        if (slash >= 0) {
            String id = ref.substring(0, slash);
            String version = ref.substring(slash + 1);
            return new GroupReference(id, version);
        } else {
            return new GroupReference(ref);
        }
    }

    /**
     * @param id
     */
    public GroupReference(String id) {
        this(id, null);
    }

    /**
     * @param id
     * @param version
     */
    public GroupReference(String id, String version) {
        if (id == null)
            throw new NullPointerException("null feature bundle id");
        this.id = id;
        this.version = version;
    }

    /**
     * @return
     */
    public String getId() {
        return id;
    }

    /**
     * @return
     */
    public String getVersion() {
        return version;
    }

    @Override
    public String toString() {
        return version != null ? id + "/" + version : id;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + id.hashCode();
        result = prime * result + ((version == null) ? 0 : version.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        GroupReference other = (GroupReference) obj;
        if (!id.equals(other.id))
            return false;
        if (version == null) {
            if (other.version != null)
                return false;
        } else if (!version.equals(other.version))
            return false;
        return true;
    }

    /**
     * @param obj group reference whose id to compare to this reference's
     * @return <code>true</code> if id's are equal
     */
    public boolean idEquals(GroupReference other) {
        return id.equals(other.id);
    }

    /**
     * Checks whether this feature reference fulfills the demand imposed by the
     * specified feature reference. Fulfillment implies matching feature ID and
     * a version match or a version range for this feature reference that includes the
     * absolute version in the specified feature reference.
     * 
     * @param ref
     * @return
     */
    public boolean includes(GroupReference request) {
        if (!id.equals(request.id))
            return false;

        if (version == null)
            // Any version will do.
            return true;

        // TODO: implement version checking

        // ASSERT: request.version is an absolute version, not a range

        // If this.version is a absolute version, check that request.version is
        // greater or equal than this.version.

        // If this.version is a version range, check that the range includes
        // request.version.

        return true;
    }

    @Override
    public int compareTo(GroupReference o) {
        int idDelta = id.compareTo(o.id);
        if (idDelta != 0)
            return idDelta;

        // TODO: comparability based on version comparability

        return 0;
    }

}
