/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.mapping;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.ObjectsWithType;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.layer0.Layer0;
import org.simantics.layer0.utils.triggers.IModification;
import org.simantics.layer0.utils.triggers.Trigger;
import org.simantics.modeling.ModelingResources;
import org.simantics.modeling.flags.LiftFlag;
import org.simantics.modeling.flags.LiftFlag.LiftedConnectionPoint;
import org.simantics.operation.Layer0X;
import org.simantics.structural.stubs.StructuralResource2;
import org.simantics.utils.ObjectUtils;

/**
 * This trigger is used for configuration diagrams of structural component types
 * to keep them in sync with the rest of the component type configuration,
 * external interface and symbol(s).
 * 
 * <p>
 * This trigger is needed because generic modifications made to component type
 * configurations do not automatically reflect to the symbol and interface.
 * Putting this logic into a trigger keeps it nicely decoupled from the normal
 * diagram editing logic.
 * 
 * @author Tuukka Lehtonen
 */
public class ComponentTypeUpdater extends Trigger {

    private static final boolean  DEBUG = false;

    protected Layer0              L0;
    protected Layer0X             L0X;
    protected DiagramResource     DIA;
    protected ModelingResources   MOD;
    protected StructuralResource2 STR;

    protected Session             session;

    private Resource              configurationDiagram;
    private Resource              configuration;
    private Resource              componentType;

    public ComponentTypeUpdater(ReadGraph g, Resource mapping) throws DatabaseException {
        L0 = Layer0.getInstance(g);
        L0X = Layer0X.getInstance(g);
        DIA = DiagramResource.getInstance(g);
        MOD = ModelingResources.getInstance(g);
        STR = StructuralResource2.getInstance(g);

        this.session = g.getSession();

        this.configurationDiagram = g.getPossibleObject(mapping, g.getInverse(L0X.HasTrigger));
        if (configurationDiagram != null) {
            this.configuration = g.getPossibleObject(configurationDiagram, MOD.DiagramToComposite);
            if (configuration != null) {
                this.componentType = g.getPossibleObject(configuration, STR.Defines);
            }
        }
    }

    @Override
    public boolean equals(Object other) {
        if (this == other)
            return true;
        if (!(other instanceof ComponentTypeUpdater))
            return false;
        ComponentTypeUpdater map = (ComponentTypeUpdater)other;
        return ObjectUtils.objectEquals(map.configurationDiagram, configurationDiagram)
                && ObjectUtils.objectEquals(map.configuration, configuration)
                && ObjectUtils.objectEquals(map.componentType, componentType);
    }

    @Override
    public int hashCode() {
        return ObjectUtils.hashCode(configurationDiagram)
                + 31 * (ObjectUtils.hashCode(configuration)
                        + 31 * ObjectUtils.hashCode(componentType));
    }

    @Override
    public IModification perform(ReadGraph graph) throws DatabaseException {
        if (DEBUG)
            System.out.println(this + ": Find modification (configuration diagram=" + configurationDiagram
                    + ", configuration=" + configuration + ", componentType=" + componentType + ")");

        if (componentType== null || configuration == null || configurationDiagram == null)
            return null;

        // Disabled because we fixed modeling rules to prevent making different
        // types of (invalid) connections to the once-created interface terminal
        // flag.

//        for (Resource connectionPoint : graph.syncRequest(new ObjectsWithType(componentType, L0.DomainOf,
//                STR.ConnectionRelation))) {
//            for (Resource flag : graph.getObjects(connectionPoint, DIA.Lifts)) {
//                LiftedConnectionPoint lcp = LiftFlag.calculateLiftedConnectionPointForFlag(graph, flag);
//                if (DEBUG)
//                    System.out.println("lifted cp(" + NameUtils.getSafeName(graph, connectionPoint, true) + "): " + lcp);
//                if (LiftFlag.isConnectionPointValid(graph, flag, connectionPoint, lcp) != null)
//                    return new LiftedConnectionPointFixer(componentType);
//            }
//        }

        return null;
    }

    /**
     * A modification that tries to fix all the connection point definitions of
     * the specified component type to match against the current connectivity of
     * the flags lifted as these connection points.
     */
    public class LiftedConnectionPointFixer implements IModification {

        private Resource componentType;

        public LiftedConnectionPointFixer(Resource componentType) {
            this.componentType = componentType;
        }

        @Override
        public void perform(WriteGraph graph) throws DatabaseException {
            for (Resource connectionPoint : graph.syncRequest(new ObjectsWithType(componentType, L0.DomainOf,
                    STR.ConnectionRelation))) {
                for (Resource flag : graph.getObjects(connectionPoint, DIA.Lifts)) {
                    LiftedConnectionPoint lcp = LiftFlag.calculateLiftedConnectionPointForFlag(graph, flag);
                    if (DEBUG)
                        System.out.println("test lifted cp(" + NameUtils.getSafeName(graph, connectionPoint, true)
                                + "): " + lcp);
                    String error = LiftFlag.isConnectionPointValid(graph, componentType, connectionPoint, lcp);
                    if (error == null)
                        continue;

                    // Fix connection point and symbol terminal type
                    if (DEBUG)
                        System.out.println("modi lifted cp(" + NameUtils.getSafeName(graph, connectionPoint, true)
                                + "): " + lcp + "\nerrors: " + error);
                    error = LiftFlag.validateConnectionPoint(graph, componentType, connectionPoint, lcp);
                    if (error != null) {
                        System.out.println(getClass().getSimpleName() + " error: " + error);
                    }
                }
            }
        }

    }

}
