/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.message.internal;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.ListenerList;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Status;
import org.simantics.message.ILogListener;
import org.simantics.message.ILogger;

/**
 * A logger for IStatus instances.
 * 
 * <p>
 * Meant to be used by any Simantics UI's for logging messages that should be
 * relevant to the user's normal workflow instead of the kinds of
 * code-error-like messages that go into the normal eclipse Error Log.
 * </p>
 * 
 * <p>
 * Use {@link #getDefault()} to get a hold of the singleton instance of Messages
 * or even more directly, you can use {@link #defaultLog(IStatus)} to
 * effectively perform <code>getDefault().log(status)</code>.
 * </p>
 * 
 * <p>
 * Originally copied from <code>org.eclipse.core.internal.runtime.Log</code>
 * since the code was internal to eclipse and not usable as such. Modifications
 * have been made since.
 * </p>
 * 
 * @author Tuukka Lehtonen
 */
final class Messages implements ILogger {

    /**
     * The log listeners.
     */
    ListenerList  listeners = new ListenerList();

    /**
     * A queue of messages that have been received before the first listener is
     * added.
     */
    List<IStatus> queue = new ArrayList<IStatus>();

    @Override
    public void addLogListener(ILogListener listener) {
        synchronized (listeners) {
            boolean firstListener = listeners.isEmpty();
            listeners.add(listener);
            if (firstListener) {
                for (IStatus s : queue) {
                    listener.logging(s, Activator.PLUGIN_ID);
                }
                queue.clear();
            }
        }
    }

    @Override
    public String getName() {
        return "Platform Log";
    }

    @Override
    public void log(IStatus status) {
        Object[] ls;
        // This synchronizes properly with addLogListener.
        synchronized (listeners) {
            ls = listeners.getListeners();
        }
        if (ls.length == 0) {
            queue.add(status);
            return;
        }
        for (Object l : ls) {
            try {
                ((ILogListener) l).logging(status, Activator.PLUGIN_ID);
            } catch (Exception e) {
                handleException(e);
            } catch (LinkageError e) {
                handleException(e);
            }
        }
    }

    private void handleException(Throwable e) {
        if (!(e instanceof OperationCanceledException)) {
            // Got an error while logging. Delegate to Error Log, which is meant
            // for errors in the code instead of these user level notifications.
            Activator.getDefault().getLog().log(new Status(IStatus.ERROR, Activator.PLUGIN_ID, "Message logging failed, see exception for cause.", e));
        }
    }

    @Override
    public void removeLogListener(ILogListener listener) {
        listeners.remove(listener);
    }

}
