/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.history;

import org.simantics.databoard.accessor.StreamAccessor;
import org.simantics.databoard.util.Bean;
import org.simantics.history.util.subscription.SubscriptionItem;

/**
 * HistoryManager is a keeper and manager of streams.
 * 
 * Note, use {@link ItemManager} for handling groups of history items. 
 * 
 * @author toni.kalajainen
 */
public interface HistoryManager {
	
	/**
	 * Create a new items. This will initialize the necessary files.
	 * A description is a record and that has at minimum the following fields:
	 * { id : String, format: Datatype }. Other records may also exist but
	 * they are not used by HistoryManager. 
	 *  
	 * If item with the same Id exists an exception is thrown.
	 * 
	 * It is recommended to use {@link SubscriptionItem} class.
	 * 
	 * @see SubscriptionItem
	 * @param descs
	 * @throws HistoryException
	 */
	void create(Bean...descs) throws HistoryException; 
	
	/**
	 * Delete a item. If an item by id does not exist, nothing happens.
	 * If there was problem deleting the item, an exception is thrown. 
	 * 	 
	 * @param itemIds item ids
	 * @throws HistoryException
	 */
	void delete(String...itemIds) throws HistoryException;
	
	/**
	 * Modify an settings of existing items. If format is changed, the
	 * implementation may thrown an exception.<p> 
	 * 
	 * If the item does not exist, the item is created.  
	 * 
	 * @param descs 
	 * @throws HistoryException
	 */
	void modify(Bean...descs) throws HistoryException;

	/**
	 * Get item description. If item does not exist, an exception is thrown.
	 * 
	 * @param itemId
	 * @return subscription info
	 * @throws HistoryException
	 */
	Bean getItem(String itemId) throws HistoryException;
	
	/**
	 * Tests if item exists in the history. 
	 * 
	 * @param itemId
	 * @return
	 * @throws HistoryException
	 */
	boolean exists(String itemId) throws HistoryException;	
			
	/**
	 * Get a snapshot of item descriptions. 
	 * 
	 * Note, ItemManager class is a higher level utility for managing the list of 
	 * items this method returns. 
	 * 
	 * @return all items
	 * @throws HistoryException
	 */
	Bean[] getItems() throws HistoryException;
	
	/**
	 * Close the history manager. Note, this does not close open streams. 
	 */
	void close();
	
	/**
	 * Open an accessor to a stream file. Time series may be opened in 
	 * read or read/write mode. Read mode may be opened even the time series
	 * is open in a collector session. Opening may fail if group operation is 
	 * underway (modify/delete).
	 * 
	 * The accessor must be closed after use.
	 * 
	 * Note, if the data is changed due to an open session, the changes may not
	 * reflect to the stream accessor. You need to {@link StreamAccessor#reset()}
	 * the accessor to flush internal caches.
	 * 
	 * @param itemId
	 * @param mode either "r" or "rw"
	 * @return accessor to stream samples.
	 * @throws HistoryException
	 */
	StreamAccessor openStream(String itemId, String mode) throws HistoryException;

}
