package org.simantics.history.examples;

import java.util.List;
import java.util.UUID;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.Datatypes;
import org.simantics.databoard.accessor.StreamAccessor;
import org.simantics.databoard.accessor.error.AccessorException;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.RecordBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.type.RecordType;
import org.simantics.databoard.util.Bean;
import org.simantics.history.History;
import org.simantics.history.HistoryException;
import org.simantics.history.HistoryItem;
import org.simantics.history.HistoryManager;
import org.simantics.history.ItemManager;

public class HistoryExample1 {

	public static void main(String[] args) throws HistoryException, BindingException, AccessorException {
		
		/// Example1: Create History Manager & Item
		// Create History Manager
		HistoryManager history = History.createMemoryHistory();
		
		// Create a sample format
		RecordType sampleFormat = new RecordType();
		sampleFormat.addComponent("time", Datatypes.DOUBLE );
		sampleFormat.addComponent("value", Datatypes.DOUBLE );
		sampleFormat.addComponent("quality", Datatypes.BYTE );
		
		// Create History Item configuration
		String id = UUID.randomUUID().toString();
		HistoryItem hi = new HistoryItem(id, sampleFormat);
		
		// Create the item in the history
		history.create( hi );
		
		/// Example 2: Write data to history item's stream (without Collector)
		{
			// Open item for writing
			StreamAccessor sa = history.openStream(id, "rw");
			// Create bean based binding for the sample
			RecordBinding sampleBinding = (RecordBinding) Bindings.getBeanBinding( sampleFormat );
			// Create sample object
			Bean sample = (Bean) sampleBinding.createPartial();
			try {
				
				// Set quality
				sample.setField(2, Bindings.BYTE, (byte)0);
				
				// Write 10 samples
				for (int i=0; i<10; i++) {
					
					// Set time
					sample.setField(0, Bindings.DOUBLE, 1.0*i );
					// Set Value
					sample.setField(1, Bindings.DOUBLE, 100.0*i );
					
					sa.add(sampleBinding, sample);				
				}
			} finally {
				sa.flush();
				sa.close();
			}
		}
		
		/// Example 3: Read & print data
		{
			// Open stream from history
			StreamAccessor aa = history.openStream(id, "r");
			// Read sample type
			RecordType sampleType = (RecordType) aa.type().componentType;
			// Create bean binding
			RecordBinding sampleBinding = (RecordBinding) Bindings.getBeanBinding( sampleType );
			// Create sample object
			Bean sample = (Bean) sampleBinding.createPartial();
			
			try {
				for (int i=0; i<aa.size(); i++) {
					aa.get(i, sampleBinding, sample);
					System.out.println(i+": "+sample);
				}
			} finally {
				aa.close();
			}
		}
		
		/// Example 4: To search and read history items
		// Read items from history. Each item is a record with various fields.
		// The fields depending on usage and initialization.
		//
		// If collector is used, it adds a field called "collectorState" that persists
		// the state data and subscription parameters. 
		Bean[] items = history.getItems();
		
		// ItemManager is a utility that helps in the management of items 
		ItemManager im = new ItemManager( items );
			
		// Print all item Ids
		System.out.println("Items Identifiers:");
		for ( String itemId : im.toIdArray() )
		{
			System.out.println( itemId );
		}
		
		// To search an item by id, use
		List<Bean> result = im.search("id", id);
		
		// To search items of specific parameter, say file format, use
		im.search("format", sampleFormat);
		
		// Or interval
		im.search("interval", 0.1);
		
	}
	
}
