package org.simantics.graph.db;

import java.util.Map;

import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.WriteOnlyGraph;
import org.simantics.db.common.procedure.adapter.TransientCacheAsyncListener;
import org.simantics.db.common.uri.UnescapedChildMapOfResource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ResourceNotFoundException;
import org.simantics.db.service.SerialisationSupport;
import org.simantics.graph.representation.External;
import org.simantics.graph.representation.Identity;
import org.simantics.graph.representation.IdentityDefinition;
import org.simantics.graph.representation.Internal;
import org.simantics.graph.representation.Root;
import org.simantics.graph.representation.TransferableGraph1;
import org.simantics.graph.representation.Value;

public class TGToGraphMap {
	private TransferableGraph1 tg;
	private Resource[] resources;
	
	private Resource RootLibrary;
	private Resource String;
	//private Resource Library;
	
	private Resource InstanceOf;
	private Resource ConsistsOf;
	private Resource PartOf;
	private Resource HasName;
	private Resource NameOf;
	
	private void findBuiltins(WriteOnlyGraph g) throws DatabaseException {
		RootLibrary = g.getBuiltin("http:/");
		String = g.getBuiltin(CoreInitialization.LAYER0 + "String");
		//Library = g.getBuiltin(CoreInitialization.LAYER0 + "Library");
		InstanceOf = g.getBuiltin(CoreInitialization.LAYER0 + "InstanceOf");
		ConsistsOf = g.getBuiltin(CoreInitialization.LAYER0 + "ConsistsOf");
		PartOf = g.getBuiltin(CoreInitialization.LAYER0 + "PartOf");
		HasName = g.getBuiltin(CoreInitialization.LAYER0 + "HasName");
		NameOf = g.getBuiltin(CoreInitialization.LAYER0 + "NameOf");
	}
	
	private void findBuiltins(ReadGraph g) throws DatabaseException {
		RootLibrary = g.getBuiltin("http:/");
		String = g.getBuiltin(CoreInitialization.LAYER0 + "String");
		//Library = g.getBuiltin(CoreInitialization.LAYER0 + "Library");
		InstanceOf = g.getBuiltin(CoreInitialization.LAYER0 + "InstanceOf");
		ConsistsOf = g.getBuiltin(CoreInitialization.LAYER0 + "ConsistsOf");
		PartOf = g.getBuiltin(CoreInitialization.LAYER0 + "PartOf");
		HasName = g.getBuiltin(CoreInitialization.LAYER0 + "HasName");
		NameOf = g.getBuiltin(CoreInitialization.LAYER0 + "NameOf");
	}
	
	public TGToGraphMap(TransferableGraph1 tg) {
		this.tg = tg;
		this.resources = new Resource[tg.resourceCount];
	}	
	
	public long[] getResources(SerialisationSupport serializer) throws DatabaseException {
		final int count = resources.length;
		long[] resourceIds = new long[count];
		for(int i=0;i<count;++i)
			resourceIds[i] = serializer.getRandomAccessId(resources[i]);
		return resourceIds;
	}	
	
	public Resource[] getResources() {
		return resources;
	}
	
	private void claimChild(WriteOnlyGraph graph, Resource parent, String name, Resource child) throws DatabaseException {
		//graph.claim(parent, ConsistsOf, PartOf, child);
		Resource nameResource = graph.newResource();
		graph.claim(nameResource, InstanceOf, null, String);
		graph.claimValue(nameResource, name, Bindings.STRING);
		graph.claim(child, HasName, NameOf, nameResource);
	}
	
	public void addMappedOldResources(
			SerialisationSupport serializer,
			int[] oldToNew,
			Resource[] oldResources) throws DatabaseException {
		assert(oldToNew.length == oldResources.length);
		for(int i=0;i<oldToNew.length;++i) {
			int id = oldToNew[i];
			if(id >= 0)
				resources[id] = oldResources[i];
		}
	}
	
	public void addOldResources(
			SerialisationSupport serializer, 
			long[] oldResources) throws DatabaseException {
		assert(oldResources.length == resources.length);
		for(int i=0;i<oldResources.length;++i) {
			resources[i] = serializer.getResource(oldResources[i]);
		}
	}

	public void prepare(ReadGraph graph) throws DatabaseException {
		if(RootLibrary == null)
			findBuiltins(graph);
		Resource[] resources = this.resources;		
		
		for(Identity identity : tg.identities) {
			IdentityDefinition definition = identity.definition;
			if(definition instanceof External) {
				External def = (External)definition;
				Resource parent = resources[def.parent];
				Resource child = graph.syncRequest(
						new UnescapedChildMapOfResource(parent), 
						new TransientCacheAsyncListener<Map<String, Resource>>())
						.get(def.name);
				if(child == null)
					throw new ResourceNotFoundException(def.name);
				resources[identity.resource] = child;
			}
			else if(definition instanceof Root) {
				resources[identity.resource] = RootLibrary;		
			}
		}
	}
	
	public void claim(WriteOnlyGraph graph) throws DatabaseException {
		if(RootLibrary == null)
			findBuiltins(graph);
		Resource[] resources = this.resources;
			
		// Create blank resources
		for(int i=0;i<resources.length;++i)
			if(resources[i] == null)
				resources[i] = graph.newResource();
		
		// Internal identities		
		for(Identity identity : tg.identities) {
			IdentityDefinition definition = identity.definition;
			if(definition instanceof Internal) {
				Internal def = (Internal)definition;
				claimChild(graph, resources[def.parent], def.name, resources[identity.resource]);
			}			
		}		
		
		// Write statements
		int[] statements = tg.statements;
		for(int i=0;i<statements.length;i+=4) {
			int inv = statements[i+2];
			graph.claim(
					resources[statements[i]],
					resources[statements[i+1]],
					inv < 0 ? null : resources[inv],
					resources[statements[i+3]]
					);
		}
				
		// Write values
		for(Value value : tg.values)
		    graph.claimValue(resources[value.resource], 
		            value.value.getValue(), value.value.getBinding());
	}
	
	public boolean checkClaim(ReadGraph graph) throws DatabaseException {
		
		if(RootLibrary == null)
			findBuiltins(graph);
		
		Resource[] resources = this.resources;
			
		// Create blank resources
		for(int i=0;i<resources.length;++i)
			if(resources[i] == null)
				return true;
		
		// Internal identities		
		for(Identity identity : tg.identities) {
			IdentityDefinition definition = identity.definition;
			if(definition instanceof Internal) {
				return true;
			}			
		}		
		
		if(tg.statements.length > 0) return true;
		if(tg.values.length > 0) return true;
		
		return false;
		
	}

	public void deny(WriteGraph graph) throws DatabaseException {
		if(RootLibrary == null)
			findBuiltins((ReadGraph)graph);
		Resource[] resources = this.resources;
		
		// Internal identities		
		for(Identity identity : tg.identities) {
			IdentityDefinition definition = identity.definition;
			if(definition instanceof Internal) {
//				Internal def = (Internal)definition;
				/*graph.deny(resources[identity.resource],
						PartOf,
						resources[def.parent]
						);*/
				graph.deny(resources[identity.resource], HasName);
			}
		}		
		
		// Deny statements
		int[] statements = tg.statements;
		for(int i=0;i<statements.length;i+=4) {
			int inv = statements[i+2];
			graph.deny(
					resources[statements[i]],
					resources[statements[i+1]],
					inv < 0 ? null : resources[inv],
					resources[statements[i+3]]
					);
		}
				
		// Deny values
		for(Value value : tg.values)
			graph.denyValue(resources[value.resource]);
	}

	public boolean checkDeny(ReadGraph graph) throws DatabaseException {
		
		if(RootLibrary == null)
			findBuiltins((ReadGraph)graph);
		
		// Internal identities		
		for(Identity identity : tg.identities) {
			IdentityDefinition definition = identity.definition;
			if(definition instanceof Internal) {
				return true;
			}
		}		
		
		if(tg.statements.length > 0) return true;
		if(tg.values.length > 0) return true;
		
		return false;
				
	}
	
}
