/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
/*
 *
 * @author Toni Kalajainen
 */
package org.simantics.g2d.participant;

import java.awt.AlphaComposite;
import java.awt.Color;
import java.awt.Composite;

import org.simantics.g2d.canvas.Hints;
import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.g2d.canvas.impl.AbstractCanvasParticipant;
import org.simantics.g2d.canvas.impl.SGNodeReflection.SGCleanup;
import org.simantics.g2d.canvas.impl.SGNodeReflection.SGInit;
import org.simantics.scenegraph.g2d.G2DParentNode;
import org.simantics.scenegraph.g2d.events.EventHandlerReflection.EventHandler;
import org.simantics.scenegraph.g2d.events.command.CommandEvent;
import org.simantics.scenegraph.g2d.events.command.Commands;
import org.simantics.scenegraph.g2d.nodes.RulerNode;
import org.simantics.utils.datastructures.hints.HintListenerAdapter;
import org.simantics.utils.datastructures.hints.IHintListener;
import org.simantics.utils.datastructures.hints.IHintObservable;
import org.simantics.utils.datastructures.hints.IHintContext.Key;
import org.simantics.utils.datastructures.hints.IHintContext.KeyOf;
import org.simantics.utils.page.MarginUtils;
import org.simantics.utils.page.MarginUtils.Margin;
import org.simantics.utils.page.MarginUtils.Margins;


public class RulerPainter extends AbstractCanvasParticipant {

    public static final int PAINT_PRIORITY = Integer.MAX_VALUE - 1000;

    public static final double RULER_WIDTH = 20.0;

    /** Ruler width + 5% */
    public static final Margin RULER_MARGIN5 = MarginUtils.marginOf(5, RulerPainter.RULER_WIDTH, 0);
    /** Ruler width + 2% */
    public static final Margin RULER_MARGIN2 = MarginUtils.marginOf(2, RulerPainter.RULER_WIDTH, 0);
    /** 5% margin + ruler */
    public static final Margins RULER_MARINGS5 = new Margins(RULER_MARGIN5, MarginUtils.MARGIN5, RULER_MARGIN5, MarginUtils.MARGIN5);
    /** 2% margin + ruler */
    public static final Margins RULER_MARINGS2 = new Margins(RULER_MARGIN2, MarginUtils.MARGIN2, RULER_MARGIN2, MarginUtils.MARGIN2);

    public static final Key KEY_RULER_ENABLED = new KeyOf(Boolean.class, "RULER_ENABLED");

    /**
     * Size of ruler in normalized (non-zoomed or 100% zoom) coordinates, not screen pixel coordinates.
     */
    public static final Key KEY_RULER_SIZE = new KeyOf(Double.class, "RULER_SIZE");

    /** Background color */
    public static final Key KEY_RULER_BACKGROUND_COLOR = new KeyOf(Color.class, "RULER_BACKGROUND_COLOR");

    public static final Key KEY_RULER_TEXT_COLOR = new KeyOf(Color.class, "RULER_TEXT_COLOR");

    public static Color DEFAULT_RULER_BACKGROUND_COLOR = new Color(192, 192, 192, 192);

    public static Color DEFAULT_RULER_TEXT_COLOR = Color.BLACK;

    IHintListener hintListener = new HintListenerAdapter() {
        public void hintChanged(IHintObservable sender, Key key, Object oldValue, Object newValue) {
            ICanvasContext cc = getContext();
            if (cc != null) {
                updateNode();
                cc.getContentContext().setDirty();
            }
        }
    };

    @Override
    public void addedToContext(ICanvasContext ctx) {
        super.addedToContext(ctx);

        if (!hasHint(KEY_RULER_SIZE))
            setHint(KEY_RULER_SIZE, RULER_WIDTH);

        getHintStack().addKeyHintListener(getThread(), KEY_RULER_ENABLED, hintListener);
        getHintStack().addKeyHintListener(getThread(), KEY_RULER_SIZE, hintListener);
        getHintStack().addKeyHintListener(getThread(), KEY_RULER_BACKGROUND_COLOR, hintListener);
        getHintStack().addKeyHintListener(getThread(), KEY_RULER_TEXT_COLOR, hintListener);
        getHintStack().addKeyHintListener(getThread(), GridPainter.KEY_GRID_SIZE, hintListener);
        getHintStack().addKeyHintListener(getThread(), Hints.KEY_DISABLE_PAINTING, hintListener);
    }

    @Override
    public void removedFromContext(ICanvasContext ctx) {
        getHintStack().removeKeyHintListener(getThread(), KEY_RULER_ENABLED, hintListener);
        getHintStack().removeKeyHintListener(getThread(), KEY_RULER_SIZE, hintListener);
        getHintStack().removeKeyHintListener(getThread(), KEY_RULER_BACKGROUND_COLOR, hintListener);
        getHintStack().removeKeyHintListener(getThread(), KEY_RULER_TEXT_COLOR, hintListener);
        getHintStack().removeKeyHintListener(getThread(), GridPainter.KEY_GRID_SIZE, hintListener);
        getHintStack().removeKeyHintListener(getThread(), Hints.KEY_DISABLE_PAINTING, hintListener);
        super.removedFromContext(ctx);
    }

    @SuppressWarnings("unused")
    private final Composite transparency = AlphaComposite.SrcOver.derive(0.75f);

    protected RulerNode node = null;

    @SGInit
    public void initSG(G2DParentNode parent) {
        node = parent.addNode("ruler", getNodeClass());
        node.setZIndex(PAINT_PRIORITY);
        updateNode();
    }

    protected Class<? extends RulerNode> getNodeClass() {
        return RulerNode.class;
    }

    @SGCleanup
    public void cleanupSG() {
        node.remove();
    }

    protected void updateNode() {
        node.setEnabled(isPaintingEnabled());
        node.setGridSize(getGridSize());
        node.setRulerSize(getRulerSize());
    }

    private double getGridSize() {
        Double d = getHint(GridPainter.KEY_GRID_SIZE);
        return d != null ? d : 1.0;
    }

    boolean isPaintingEnabled()
    {
        boolean enabled = isEnabled();
        Boolean globalDisable = getHint(Hints.KEY_DISABLE_PAINTING);
        return enabled && !Boolean.TRUE.equals(globalDisable);
    }

    public boolean isEnabled()
    {
        Boolean b = getHint(KEY_RULER_ENABLED);
        return !Boolean.FALSE.equals(b);
    }

    private double getRulerSize() {
        Double d = getHint(KEY_RULER_SIZE);
        return d != null ? d : RULER_WIDTH;
    }

    public Color getRulerTextColor()
    {
        Color c = getHint(KEY_RULER_TEXT_COLOR);
        if (c!=null) return c;
        return DEFAULT_RULER_TEXT_COLOR;
    }

    public Color getRulerBackgroundColor()
    {
        Color c = getHint(KEY_RULER_BACKGROUND_COLOR);
        if (c!=null) return c;
        return DEFAULT_RULER_BACKGROUND_COLOR;
    }

    public void setEnabled(boolean enabled)
    {
        node.setEnabled(enabled);
        setHint(KEY_RULER_ENABLED, enabled);
    }

    public void setRulerTextColor(Color c)
    {
        setHint(KEY_RULER_TEXT_COLOR, c);
    }

    public void setRulerBackgroundColor(Color c)
    {
        setHint(KEY_RULER_BACKGROUND_COLOR, c);
    }

    @EventHandler(priority = 0)
    public boolean handleKeyEvent(CommandEvent e) {
        if (e.command.equals( Commands.RULER_ENABLE)) {
            setEnabled(true);
            return true;
        } else if (e.command.equals( Commands.RULER_DISABLE)) {
            setEnabled(false);
            return true;
        } else if (e.command.equals( Commands.RULER_TOGGLE)) {
            setEnabled(!isEnabled());
            return true;
        }
        return false;
    }

}
