/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.g2d.image.impl;

import java.awt.geom.AffineTransform;
import java.util.EnumSet;

import org.simantics.g2d.image.Image;
import org.simantics.g2d.image.ImageUtils;
import org.simantics.scenegraph.Node;
import org.simantics.scenegraph.g2d.G2DParentNode;

/**
 * System-memory stored raster of an image.
 * Suitable for vector based images.
 * <p>
 * This implementation determines the required raster quality from the
 * given graphics2D context. If the buffered raster is not good enough quality,
 * the image will be re-rendered. If the buffer to be rendered is larger than
 * maxDimension * maxDimension, original image will be used instead.
 * <p>
 * Disposing buffer does not dipose the original image.
 * 
 * @See {@link ImageUtils}
 * @author Toni Kalajainen
 */
public class BufferedImage extends ImageProxy implements Image {

    public static final double MAX_DIMENSION = 800;

    java.awt.image.BufferedImage raster;
    double rasterResolution = 1.0;
    double maxDimension;
    EnumSet<Feature> caps;

    public BufferedImage(Image source)
    {
        this(source, MAX_DIMENSION);
    }

    public BufferedImage(Image source, double maxDimension)
    {
        super(source);
        this.maxDimension = maxDimension;
        boolean vector = source.getFeatures().contains(Feature.Vector);
        boolean vola = source.getFeatures().contains(Feature.Volatile);
        if (vola) {
            source.addImageListener(new ImageListener() {
                @Override
                public void onContentChangedNotification(Image image) {
                    releaseRaster();
                }});
            this.caps = EnumSet.of(Feature.Volatile);
        } else {
            this.caps = EnumSet.noneOf(Feature.class);
        }
    }

    private double requiredResolution(AffineTransform af)
    {
        double reso = Math.max(af.getScaleX(), af.getScaleY());
        // Add some extra information for rotations & anti-aliasing
        reso *= 1.3;
        return reso;
    }

    private synchronized void releaseRaster()
    {
        raster = null;
    }

    @Override
    public synchronized Node init(G2DParentNode parent) {
        return null;
//		Graphics2D g = gc.getGraphics2D();
//		// Quality rendering requested, do not render from cache
//		if (g.getRenderingHint(RenderingHints.KEY_RENDERING) == RenderingHints.VALUE_RENDER_QUALITY)
//		{
//			source.paint(gc);
//			return;
//		}
//		Rectangle2D bounds = getBounds();
//
//		double requiredResolution = requiredResolution(g.getTransform());
//		double wid = bounds.getWidth();
//		double hei = bounds.getHeight();
//		wid *= requiredResolution * 1.05;
//		hei *= requiredResolution * 1.05;
//
//		// too detailed, render with the original image
//		if (wid*hei > maxDimension*maxDimension)
//		{
//			source.paint(gc);
//			return;
//		}
//
//		if (raster!=null && requiredResolution>rasterResolution )
//		{
//			releaseRaster();
//		}
//		if (raster==null) {
//
//			rasterResolution = requiredResolution;
//
//			raster = new java.awt.image.BufferedImage(
//					(int)(wid+1),
//					(int)(hei+1),
//					java.awt.image.BufferedImage.TYPE_INT_ARGB);
//			Graphics2D target = raster.createGraphics();
//			target.setBackground(new Color(255,255,255,0));
//			target.clearRect(0, 0, raster.getWidth(), raster.getHeight());
//			target.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
//			target.scale(rasterResolution, rasterResolution);
//			target.translate(-bounds.getX(), -bounds.getY());
//
//			QualityHints.HIGH_QUALITY_HINTS.setQuality(target);
//			target.setRenderingHint( RenderingHintsKeyExt.KEY_BUFFERED_IMAGE, new WeakReference<java.awt.image.BufferedImage>(raster) );
//
//			source.paint(new GraphicsContextImpl(new Rectangle2D.Double(0,0, raster.getWidth(), raster.getHeight()), null)  );
//			target.dispose();
//		}
//
//		AffineTransform af = g.getTransform();
//		try {
//			g.translate(bounds.getX(), bounds.getY());
//			g.scale(1/rasterResolution, 1/rasterResolution);
//			g.drawImage(raster, 0, 0, null);
//			g.setColor(Color.BLACK);
//		} finally {
//			g.setTransform(af);
//		}
    }

    @Override
    public EnumSet<Feature> getFeatures() {
        return caps;
    }

}

