/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.fastlz;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;

import org.simantics.fastlz.java.FastLZImpl;
import org.simantics.fastlz.java.FastLZJavaInputStream;
import org.simantics.fastlz.java.FastLZJavaOutputStream;

/**
 * A Java port of the native {@link FastLZ} library.
 * 
 * @author Tuukka Lehtonen
 * 
 * @see FastLZ
 * @see FastLZImpl
 * @see FastLZJavaInputStream
 * @see FastLZJavaOutputStream
 */
public class FastLZJava {

    /**
     * Compress a block of data in the input buffer and returns the size of
     * compressed block. The size of input buffer is specified by length. The
     * minimum input buffer size is 16.
     * 
     * <p>
     * The output buffer must be at least 5% larger than the input buffer and
     * can not be smaller than 66 bytes.
     * 
     * <p>
     * If the input is not compressible, the return value might be larger than
     * length (input buffer size).
     * 
     * <p>
     * The input buffer and the output buffer can not overlap.
     */
    public static int compress(byte[] input, int inputOffset, int length, byte[] output, int outputOffset) {
        return FastLZImpl.fastlz_compress(input, inputOffset, length, output, outputOffset);
    }

    /**
     * Decompress a block of compressed data and returns the size of the
     * decompressed block. If error occurs, e.g. the compressed data is
     * corrupted or the output buffer is not large enough, then 0 (zero) will be
     * returned instead.
     * 
     * <p>
     * The input buffer and the output buffer can not overlap.
     * 
     * <p>
     * Decompression is memory safe and guaranteed not to write the output
     * buffer more than what is specified in maxout.
     */
    public static int decompress(byte[] input, int inputOffset, int length, byte[] output, int outputOffset, int maxout) {
        return FastLZImpl.fastlz_decompress(input, inputOffset, length, output, outputOffset, maxout);
    }

    /**
     * @param file the FastLZ-compressed file to read
     * @return input stream that decompresses its output using the FastLZ
     *         algorithm
     * @throws FileNotFoundException see
     *         {@link FileOutputStream#FileOutputStream(File)} for when this is
     *         thrown
     */
    public static InputStream read(File file) throws FileNotFoundException {
        return new FastLZJavaInputStream(new FileInputStream(file));
    }

    /**
     * @param file the FastLZ-compressed file to write
     * @return output stream that compresses its input using the FastLZ
     *         algorithm
     * @throws FileNotFoundException see
     *         {@link FileOutputStream#FileOutputStream(File)} for when this is
     *         thrown
     */
    public static OutputStream write(File file) throws FileNotFoundException {
        return new FastLZJavaOutputStream(new FileOutputStream(file));
    }

}
