/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.editors;

import java.net.URL;

import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.SWT;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IPersistableElement;

/**
 * EditorInput for the Browser component.
 * 
 * 
 * @author Marko Luukkainen <Marko.Luukkainen@vtt.fi>
 * @author Tuukka Lehtonen
 */
public class BrowserInput implements IEditorInput {

    private URL           url;
    private String        name;
    private final boolean showControls;
    private final boolean fullscreen;
    private int           browserStyle = SWT.NONE;

    /**
     * Creates an input for a browser that has visible controls.
     * 
     * @param url the url that browser opens.
     */
    public BrowserInput(URL url) {
        this.url = url;
        this.showControls = true;
        this.fullscreen = false;
    }

    /**
     * Creates an input for a browser that has visible controls.
     * 
     * @param url the url that browser opens.
     */
    public BrowserInput(URL url, String name) {
        this.url = url;
        this.showControls = true;
        this.fullscreen = false;
    }

    /**
     * Creates an input for a browser.
     * 
     * @param url the url that browser opens.
     * @param showControls visibility flag of controls
     */
    public BrowserInput(URL url, boolean showControls) {
        this.url = url;
        this.showControls = showControls;
        this.fullscreen = false;
    }

    public BrowserInput(URL url, boolean showControls, boolean fullscreen) {
        this.url = url;
        this.showControls = showControls;
        this.fullscreen = fullscreen;
    }

    public BrowserInput(URL url, boolean showControls, boolean fullscreen, int style) {
        this(url, null, showControls, fullscreen, style);
    }

    public BrowserInput(URL url, String name, boolean showControls, boolean fullscreen, int style) {
        this.url = url;
        this.name = name;
        this.showControls = showControls;
        this.fullscreen = fullscreen;
        this.browserStyle = style;
    }

    public void setUrl(URL url) {
        this.url = url;
    }

    public URL getUrl() {
        return url;
    }

    public boolean isShowControls() {
        return showControls;
    }

    @Override
    public boolean exists() {
        return false;
    }

    @Override
    public ImageDescriptor getImageDescriptor() {
        return null;
    }

    public void setName(String name) {
        this.name = name;
    }

    @Override
    public String getName() {
        if (name != null)
            return name;
        return url.toString();
    }

    @Override
    public IPersistableElement getPersistable() {
        return null;
    }

    @Override
    public String getToolTipText() {
        //return getName();
        return getUrl().toString();
    }

    public boolean isFullscreen() {
        return fullscreen;
    }

    public int getBrowserStyle() {
        return browserStyle;
    }

    @SuppressWarnings({ "rawtypes" })
    @Override
    public Object getAdapter(Class adapter) {
        return null;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + browserStyle;
        result = prime * result + (fullscreen ? 1231 : 1237);
        result = prime * result + (showControls ? 1231 : 1237);
        result = prime * result + ((url == null) ? 0 : url.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        BrowserInput other = (BrowserInput) obj;
        if (browserStyle != other.browserStyle)
            return false;
        if (fullscreen != other.fullscreen)
            return false;
        if (showControls != other.showControls)
            return false;
        if (url == null) {
            if (other.url != null)
                return false;
        } else if (!url.equals(other.url))
            return false;
        return true;
    }

}
