/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.document;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;

import org.eclipse.mylyn.wikitext.mediawiki.MediaWikiLanguage;
import org.eclipse.mylyn.wikitext.parser.MarkupParser;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.utils.Logger;
import org.simantics.db.exception.DatabaseException;
import org.simantics.utils.FileUtils;
import org.simantics.utils.ui.BundleUtils;
import org.simantics.wiki.ui.editor.IExportable;

import com.lowagie.text.Document;
import com.lowagie.text.DocumentException;
import com.lowagie.text.PageSize;
import com.lowagie.text.Utilities;
import com.lowagie.text.pdf.PdfContentByte;
import com.lowagie.text.pdf.PdfImportedPage;
import com.lowagie.text.pdf.PdfReader;
import com.lowagie.text.pdf.PdfWriter;

public class Exportable implements IExportable {

	final private String html;
	final private DocumentSettings settings;

	private static final String DEFAULT_CSS;
	public static final String DEFAULT_MODEL_CSS;

	private static String getBundleFileContents(String path, String defaultValue) {
		URL url = BundleUtils.find(Activator.getContext().getBundle(), path);
		if (url == null)
			return defaultValue;
		try (InputStream in = url.openStream()) {
			return FileUtils.getContents(in);
		} catch (IOException e) {
			Logger.defaultLogError(e);
			return defaultValue;
		}
	}

	static {
		DEFAULT_CSS = getBundleFileContents("simantics-wiki-documents.css", "");
		DEFAULT_MODEL_CSS = getBundleFileContents("simantics-wiki-documents-default-model.css", "");
	}
	
	public Exportable(ReadGraph graph, Resource res, String wiki, String css, DocumentSettings settings, boolean print) {

		try {
			wiki = DocumentDialect.INSTANCE.apply(graph, res, wiki);
		} catch (DatabaseException e) {
			Logger.defaultLogError(e);
		}
		
		this.settings = settings;
		
		MarkupParser markupParser = new MarkupParser();
		markupParser.setMarkupLanguage(new MediaWikiLanguage());
		String html = markupParser.parseToHtml(wiki);
		
		String width = "width:" + (210-settings.marginLeft-settings.marginRight) + "mm;";

		if(print) {
			html = html.replace("<body>", "<body style=\"background-color:#fff\"><div style=\"background-color:#FFF;" + width + "\">");
			html = html.replace("</body>", "</div></body>");
		} else {
			String div1 = "margin-left:3mm;margin-top:3mm;background-color:#FFF;width:210mm;"; 
			String div2 = "background-color:#FFF;width:210mm;padding-top:" + settings.marginTop + "mm;";
			String div3 = "overflow-x:hidden;margin-left:" + settings.marginLeft + "mm;background-color:#FFF;" + width; 
			html = html.replace("<body>", "<body><div style=\"" + div1 + "\"><div style=\"" + div2 + "\"><div style=\"" + div3 + "\">");
			html = html.replace("</body>", "</div></div></div></body>");
		}
		
		html = html.replace("<td>", "<td><div>");
		html = html.replace("<th>", "<th><div>");
		html = html.replace("</td>", "</div></td>");
		html = html.replace("</th>", "</div></th>");
		html = html.replace("</head>", "<style type=\"text/css\">\n" + DEFAULT_CSS + css +  "\n</style>\n</head>\n");
		
		this.html = html;
		
	}
	
	public String getHTML() {
		return html;
	}

	@Override
	public int export(Document document, PdfWriter writer) throws DocumentException {

		File temp = Simantics.getTempfile("wikiPdfExport", "pdf");
		try {
			
			temp.getParentFile().mkdirs();
			PhantomJSDriver.print(html, settings, temp);
			
			int result = 0;
	        PdfContentByte cb = writer.getDirectContent();
	        PdfReader reader = new PdfReader(new BufferedInputStream(new FileInputStream(temp)));
	        for (int i = 1; i <= reader.getNumberOfPages(); i++) {
	        	document.setPageSize(PageSize.A4);
	        	document.newPage();
	        	//import the page from source pdf
	        	PdfImportedPage page = writer.getImportedPage(reader, i);
	        	//add the page to the destination pdf
	        	float pts = Utilities.millimetersToPoints(10);
	        	cb.addTemplate(page, pts, pts);
	        	++result;
	        }

	        return result;
		} catch (IOException e) {

			throw new DocumentException(e);
			
		}

	}

}