/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.synchronization;

import java.util.EnumSet;
import java.util.Map;

import org.simantics.g2d.diagram.IDiagram;

/**
 * A CopyAdvisor is used to abstractly evaluate whether a source object can be
 * copied and for performing the actual copy operation. CopyAdvisor is a part of
 * the Simantics diagram to backend (graph) synchronization framework.
 * 
 * <p>
 * The {@link #canCopy(ISynchronizationContext, Object)} method returns an
 * evaluation of what can be regarding copying of the specified source object.
 * The type of the source object depends on the backend just like the contents
 * of the received {@link ISynchronizationContext}.
 * 
 * <p>
 * CopyAdvisor is currently used with {@link IDiagram} to describe how copying
 * happens within a diagram (or diagram type).
 * 
 * @author Tuukka Lehtonen
 */
public interface CopyAdvisor {

    public enum Evaluation {
        SUPPORTED,
        NOT_SUPPORTED,
        DENIED,
    }

    EnumSet<Evaluation> SUPPORTED  = EnumSet.of(Evaluation.SUPPORTED);
    EnumSet<Evaluation> DENIED     = EnumSet.of(Evaluation.DENIED);
    EnumSet<Evaluation> NOT_DENIED = EnumSet.complementOf(DENIED);

    /**
     * @param context a context for the synchronization operation
     * @param source
     * @param sourceContainer
     * @param targetContainer
     * @return evaluation of the possibilities of copying the source object
     */
    Evaluation canCopy(ISynchronizationContext context, Object source, Object sourceContainer, Object targetContainer);

    /**
     * @param context a context for the synchronization operation
     * @param source
     * @param sourceContainer
     * @param targetContainer
     * @return copied backend object or <code>null</code> if the specified
     *         source object is not supported
     * @throws SynchronizationException if copying is denied for the source
     *         object
     */
    Object copy(ISynchronizationContext context, Object source, Object sourceContainer, Object targetContainer);

    /**
     * @param context a context for the synchronization operation
     * @param source
     * @param sourceContainer
     * @param targetContainer
     * @param map a map for storing the correspondences between original and
     *        copied objects. This is used to output data from the copy process.
     * @return copied backend object or <code>null</code> if the specified
     *         source object is not supported
     * @throws SynchronizationException if copying is denied for the source
     *         object
     */
    Object copy(ISynchronizationContext context, Object source, Object sourceContainer, Object targetContainer,
            Map<Object, Object> map);

    /**
     * @param context
     * @param source
     * @param sourceContainer
     * @param targetContainer
     * @return any non-null object if successful, <code>null</code> if couldn't
     *         cut
     * @throws SynchronizationException if cut fails
     */
    Object cut(ISynchronizationContext context, Object source, Object sourceContainer, Object targetContainer);

    /**
     * Invoked when either a copy or cut operation has been finished.
     * 
     * @param context
     * @throws SynchronizationException if onFinish fails
     */
    void onFinish(ISynchronizationContext context);

}
