/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.adapter;

import java.awt.geom.AffineTransform;
import java.util.Map;

import org.simantics.db.AsyncReadGraph;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.procedure.adapter.AsyncProcedureAdapter;
import org.simantics.db.common.procedure.adapter.ProcedureAdapter;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.procedure.AsyncProcedure;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.diagram.synchronization.ErrorHandler;
import org.simantics.diagram.synchronization.ISynchronizationContext;
import org.simantics.diagram.synchronization.SynchronizationHints;
import org.simantics.diagram.synchronization.graph.DiagramGraphUtil;
import org.simantics.diagram.synchronization.graph.GraphSynchronizationHints;
import org.simantics.diagram.synchronization.graph.layer.GraphLayerManager;
import org.simantics.g2d.diagram.DiagramHints;
import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.element.ElementUtils;
import org.simantics.g2d.element.IElement;
import org.simantics.g2d.layers.ILayersEditor;
import org.simantics.utils.datastructures.hints.IHintContext.Key;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ElementFactoryUtil {

    private static final Logger LOGGER = LoggerFactory.getLogger(ElementFactoryUtil.class);

    public static void readTransform(AsyncReadGraph graph, final Resource resource, final IElement e) {
        readTransform(graph, resource, e, new AsyncProcedureAdapter<IElement>() {
            @Override
            public void exception(AsyncReadGraph graph, Throwable t) {
                LOGGER.error("Error reading transform from " + resource + " into element " + e, t);
            }
        });
    }

    public static void readTransform(AsyncReadGraph graph, final Resource resource, final IElement e, final AsyncProcedure<IElement> procedure) {
        DiagramResource DIA = graph.getService(DiagramResource.class);
        try {
            AffineTransform tr = DiagramGraphUtil.getDynamicAffineTransform((ReadGraph)graph, null, resource, DIA.HasDynamicTransform, true);
            ElementUtils.setTransform(e, tr);
        } catch (DatabaseException e1) {
            ElementUtils.setTransform(e, new AffineTransform());
        }
        procedure.execute(graph, e);
    }

    public static void readParameters(AsyncReadGraph graph, final Resource resource, final IElement e) {
        graph.asyncRequest(new ElementParameterRequest(resource), new ProcedureAdapter<Map<String,Object>>() {
            @Override
            public void execute(Map<String, Object> result) {
                ElementUtils.setParameters(e, result);
            }
        });
    }

    public static ISynchronizationContext getContext(IDiagram diagram) {
        return diagram.getHint(SynchronizationHints.CONTEXT);
    }

    public static ISynchronizationContext getContextChecked(IDiagram diagram) {
        ISynchronizationContext context = getContext(diagram);
        if (context == null)
            throw new IllegalStateException("diagram does not contain " + SynchronizationHints.CONTEXT + " hint");
        return context;
    }

    public static ErrorHandler getErrorHandler(IDiagram diagram) {
        ISynchronizationContext context = getContext(diagram);
        if (context == null)
            return null;
        return context.get(SynchronizationHints.ERROR_HANDLER);
    }

    public static ErrorHandler getErrorHandlerChecked(IDiagram diagram) {
        ISynchronizationContext context = getContextChecked(diagram);
        ErrorHandler errorHandler = context.get(SynchronizationHints.ERROR_HANDLER);
        if (errorHandler == null)
            throw new IllegalStateException("synchronization context " + context + " does not contain " + SynchronizationHints.ERROR_HANDLER + " hint");
        return errorHandler;
    }

    public static <T> T getContextHint(IDiagram diagram, Key key) {
        ISynchronizationContext context = getContext(diagram);
        if (context == null)
            return null;
        return context.get(key);
    }

    public static <T> T getContextHintChecked(IDiagram diagram, Key key) {
        return getContextChecked(diagram).get(key);
    }

    public static void loadLayersForElement(AsyncReadGraph graph, IDiagram diagram, IElement e, Resource element,
            AsyncProcedure<IElement> callback) {
        final GraphLayerManager layerManager = getContextHint(diagram, GraphSynchronizationHints.GRAPH_LAYER_MANAGER);
        if (layerManager != null)
            loadLayersForElement(graph, layerManager, diagram, e, element, callback);
        else
            callback.execute(graph, e);
    }

    public static void loadLayersForElement(AsyncReadGraph graph, GraphLayerManager layerManager, IDiagram diagram, IElement e, Resource element,
            AsyncProcedure<IElement> callback) {
        final ILayersEditor layers = diagram.getHint(DiagramHints.KEY_LAYERS_EDITOR);
        if (layers != null)
            layerManager.loadLayersForElement(graph, layers, e, element, callback);
        else
            callback.execute(graph, e);
    }

}
