/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.adapter;

import org.simantics.db.AsyncReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.procedure.guarded.GuardedAsyncProcedureWrapper;
import org.simantics.db.procedure.AsyncProcedure;
import org.simantics.db.procedure.Procedure;
import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.element.ElementClass;
import org.simantics.g2d.element.IElement;

/**
 * The base interface for loading diagram elements from the graph database into
 * {@link ElementClass} and {@link IElement} instances for use with the g2d
 * diagram framework.
 * 
 * @author Tuukka Lehtonen
 * 
 * @see ElementFactoryAdapter
 * @see SyncElementFactory
 */
public interface ElementFactory {

    /**
     * Retrieves the {@link ElementClass} for the given modelled element type.
     */
    void create(AsyncReadGraph graph, ICanvasContext canvas, IDiagram diagram, Resource elementType, AsyncProcedure<ElementClass> procedure);

    /**
     * Retrieves the {@link ElementClass} for the given modelled element
     * instance. The only real difference to
     * {@link #create(AsyncReadGraph, ICanvasContext, IDiagram, Resource, AsyncProcedure)}
     * is that the resource argument is an element instance, not a type.
     * 
     * The most usual default implementation of this method is to get the type
     * of the resource and invoke
     * {@link #create(AsyncReadGraph, ICanvasContext, IDiagram, Resource, AsyncProcedure)}
     * with the type.
     */
    void getClass(AsyncReadGraph graph, ICanvasContext canvas, IDiagram diagram, Resource elementResource, AsyncProcedure<ElementClass> procedure);

    /**
     * Loads data from given modelled element resource into the given element.
     * 
     * <p>
     * The loader shall notify the specified <code>procedure</code> when the
     * element has been loaded. If the element cannot be loaded properly for
     * some reason, {@link Procedure#exception(Throwable)} should be invoked.
     * Only invoke either {@link Procedure#exception(Throwable)} or
     * {@link Procedure#execute(Object)} once.
     * {@link GuardedAsyncProceureWrapper} can help with that.
     * 
     * <p>
     * IMPORTANT: Diagram elements are essentially nothing more than their
     * {@link ElementClass} and a set of hints. Updating the appearance or any
     * other property related to an IElement is always a matter of updating its
     * hint values. This is exactly what {@link GraphToDiagramSynchronizer} does
     * in order to keep the run-time IElement representation in sync with the
     * database contents. IElements may contain any number of hints that are set
     * outside of the ElementFactory that loaded the element. In order for the
     * graph &rarr; IElement updates to work properly, the updater must be able
     * to rely on the fact that it simply needs to replace the existing hint
     * values with the new hint values loaded by this method. Therefore
     * implementers need to ensure that the same set of hint keys is always
     * loaded for a single element. Otherwise the update process will
     * potentially leave unwanted hints lying around in the run-time diagram
     * elements.
     * 
     * @see GuardedAsyncProcedureWrapper
     */
    void load(AsyncReadGraph graph, ICanvasContext canvas, IDiagram diagram, Resource elementResource, IElement element, AsyncProcedure<IElement> procedure);

}
