/*******************************************************************************
 * Copyright (c) 2016 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     THTH ry - initial API and implementation
 *******************************************************************************/
package org.simantics.debug.browser.servlet;

import java.io.IOException;
import java.io.PrintWriter;
import java.net.URI;
import java.net.URISyntaxException;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.common.request.Queries;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.common.request.UnaryRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ResourceNotFoundException;
import org.simantics.db.service.ClusteringSupport;
import org.simantics.db.service.SerialisationSupport;
import org.simantics.debug.browser.content.ResourceBrowserContent;

public class ResourceBrowserServlet extends HttpServlet {

	private static final long serialVersionUID = -8253560202827865253L;

	@Override
	protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
		response.setContentType("text/html; charset=utf-8");
		response.setStatus(HttpServletResponse.SC_OK);

		final PrintWriter writer = response.getWriter();
		try {
			Session session = Simantics.peekSession();
			if (session == null) {
				writer.write("<html><body>No database session.</body></html>");
				return;
			}

			String requestedResource = request.getPathInfo();
			if (requestedResource == null || requestedResource.equals("/")) {
				long rootId = session.getRootLibrary().getResourceId();
				//writer.write("<html><body>No resource requested. Try <a href=\"" + rootId + "\">database root</a>.</body></html>");
				response.sendRedirect("" + rootId);
				return;
			}

			// Skip '/' suffix
			requestedResource = requestedResource.substring(1);
			final long resource = parseLong(requestedResource);

			if (resource != 0L) {
				session.syncRequest(new ReadRequest() {
					@Override
					public void run(ReadGraph graph) throws DatabaseException {
						ResourceBrowserContent content = graph.syncRequest(new GetContent(resource));
						content.toHtml(graph, writer);
					}
				});
			} else {
				writer.write("<html><body>Resource not found: <em>");
				writer.write(requestedResource);
				writer.write("</em></body></html>");
			}
		} catch (Throwable e) {
			writer.write("<html><body>Problem occurred while reading <em>");
			writer.write(request.getPathInfo());
			writer.write("</em>:<br/><pre>");
			e.printStackTrace(writer);
			writer.write("</pre></body></html>");
		}
	}
	
	@Override
	protected void doPost(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
	    
	    Resource foundResource = null;
	    Session session = Simantics.getSession();
	    
	    String input = request.getParameter("resource-input");
	    if (input != null && !input.isEmpty()) {
            // There's no harm in trimming out spaces from both ends of the input.
            input = input.trim();
            SerialisationSupport support = session.getService(SerialisationSupport.class);
            if (input.startsWith("$")) {
                try {
                    foundResource = support.getResource(Long.parseLong(input.substring(1)));
                } catch (NumberFormatException e) {
                    // Let's silently fail here for now and navigate to project resource in the end
                    //e.printStackTrace();
                } catch (Exception e) {
                    // Let's silently fail here for now and navigate to project resource in the end
                    //e.printStackTrace();
                }
            }
    
            if (foundResource == null) {
                String[] parts = input.split("-");
                if (parts.length == 1) {
                    try {
                        int resourceKey = Integer.parseInt(parts[0].trim());
                        foundResource = support.getResource(resourceKey);
                        // Some validation, not enough though
                        ClusteringSupport cs = session.getService(ClusteringSupport.class);
                        long cluster = cs.getCluster(foundResource);
                        if(cluster > 0) {
        //                    changeLocation(r);
                        }
                    } catch (NumberFormatException e1) {
                        // Ignore, may happen for crap input
                        // e1.printStackTrace();
                    } catch (Exception e1) {
                        // Let's silently fail here for now and navigate to project resource in the end
                        //e1.printStackTrace();
                    }
                } else if (parts.length == 2) {
                    try {
                        int resourceIndex = Integer.parseInt(parts[1]);
                        long clusterId = Long.parseLong(parts[0]);
                        ClusteringSupport cs = session.getService(ClusteringSupport.class);
                        foundResource = cs.getResourceByIndexAndCluster(resourceIndex, clusterId);
                    } catch (NumberFormatException e1) {
                        // Let's silently fail here for now and navigate to project resource in the end
                        //e1.printStackTrace();
                    } catch (Exception e1) {
                        // Let's silently fail here for now and navigate to project resource in the end
                        //e1.printStackTrace();
                    }
                }
            }
            if (foundResource == null) {
                // Try to see if the input data is an URI reference
                try {
                    // First check that the input really is a proper URI.
                    String uri = input;
                    if (!input.equals("http:/") && input.endsWith("/"))
                        uri = input.substring(0, input.length() - 1);
                    new URI(uri);
                    foundResource = session.syncRequest( Queries.resource( uri ) );
                } catch (URISyntaxException e) {
                    // Ignore, this is not a proper URI at all.
                } catch (ResourceNotFoundException e1) {
                    // Ok, this was an URI, but no resource was found.
                    // Let's silently fail here for now and navigate to project resource in the end
                    //e1.printStackTrace();
                } catch (DatabaseException e1) {
                    // Let's silently fail here for now and navigate to project resource in the end
                    //e1.printStackTrace();
                }
            }
	    }
        
        if (foundResource != null) {
            response.sendRedirect("" + foundResource.getResourceId());
        } else {
            long rootId = session.getRootLibrary().getResourceId();
            response.sendRedirect("" + rootId);
        }
        return;
	}

	private static long parseLong(String s) {
		try {
			return Long.parseLong(s);
		} catch (NumberFormatException e) {
			return 0;
		}
	}

	static class GetContent extends UnaryRead<Long, ResourceBrowserContent> {
		public GetContent(long parameter) {
			super(parameter);
		}

		@Override
		public ResourceBrowserContent perform(ReadGraph graph) throws DatabaseException {
			Resource r = graph.getService(SerialisationSupport.class).getResource(parameter);
			return ResourceBrowserContent.createContentFor(graph, r);
		}
	}

}
