/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.service;

import java.util.Collection;

import org.simantics.db.ChangeSet;
import org.simantics.db.ChangeSetIdentifier;
import org.simantics.db.Metadata;
import org.simantics.db.ReadGraph;
import org.simantics.db.exception.DatabaseException;

public interface ManagementSupport {

    /**
     * Get change sets from server.
     * IMPORTANT WARNING:
     * Due to implementation deficiencies the returned collections
     * changesStatements and changesValues do not implement the collection
     * interface. All you can do is to iterate the collections.
     * Also note that the current interface does not offer a way to access
     * the changed values. (The implementation does not even store the changed
     * values. The used implementation for change set is ClientChangesImpl
     * because it's backed to disk and change set can be large. ClientChangeSetImpl
     * would offer a better implementation of collections but is not suitable for
     * large change sets.)
     *
     * @param graph to ensure that we have a read transaction.
     * @return ChangeSets for given inclusive range [from, to].
     * @throws DatabaseException
     */
    public Collection<ChangeSet> fetchChangeSets(ReadGraph graph, long from, long to)
            throws DatabaseException;

    /**
     * Get information about change sets in server.
     *
     * @return ChangeSetIdentifiers for given inclusive range [from, to].
     * @throws DatabaseException
     */
    Collection<ChangeSetIdentifier> getChangeSetIdentifiers(long from, long to)
            throws DatabaseException;

    /**
     * Get information about change sets in server.
     *
     * @return Metadata objects for given change set for the inclusive range
     *         [from, to].
     * @throws DatabaseException
     */
    <T> Collection<T> getMetadata(long from, long to, Class<? extends Metadata> dataClass)
            throws DatabaseException;

    /**
     * Get meta information about changes sets in server. This was implemented
     * before the {@link #getMetadata(long, long, Class)} and was left for
     * backward compatibility. Prefer {@link #getMetadata(long, long, Class)}.
     *
     * @return Metadata objects for given change set for the inclusive range
     *         [from, to].
     * @throws DatabaseException
     */
    <T> Collection<T> getMetadata(ReadGraph graph, long from, long to, Class<? extends Metadata> dataClass)
            throws DatabaseException;

    /**
     * Create and save to file selected revision in server.
     * @return
     * @throws DatabaseException
     */
    void dumpRevision(long changeSetId)
            throws DatabaseException;

    /**
     * Save change sets to file up to selected revision in server.
     * @return
     * @throws DatabaseException
     */
    void dumpChangeSets(long changeSetId)
            throws DatabaseException;

    /**
     * Get head revision id.
     * @return last revision id.
     * @throws DatabaseException
     */
    long getHeadRevisionId()  throws DatabaseException;

    /**
     * Get first revision id that is or will be available when next revision is created.
     *
     * @return first revision id.
     * @throws DatabaseException
     */
    long getFirstRevisionId() throws DatabaseException;

    interface ChangeSetListener {
        void onChanged(long headChangeSetId);
    }
    void subscribe(ChangeSetListener changeSetListener);
    void cancel(ChangeSetListener changeSetListener);

    /**
     * Get information about change sets in server. This was implemented before
     * {@link #getChangeSetIdentifiers(long, long)} and was left for backward
     * compatibility. Prefer {@link #getChangeSetIdentifiers(long, long)}.
     * This will be removed soon.
     *
     * @return ChangeSetIdentifiers for given inclusive range [from, to].
     * @throws DatabaseException
     */
    @Deprecated
    Collection<ChangeSetIdentifier> getChangeSets(long from, long to)
            throws DatabaseException;

}
