/*******************************************************************************
 * Copyright (c) 2007, 2018 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl.query;

import org.simantics.databoard.util.URIStringUtils;
import org.simantics.db.ObjectResourceIdMap;
import org.simantics.db.common.exception.DebugException;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ResourceNotFoundException;
import org.simantics.db.impl.graph.ReadGraphImpl;
import org.simantics.db.impl.procedure.InternalProcedure;

public class URIToResource extends StringQuery<InternalProcedure<Integer>> implements InternalProcedure<Integer> {

    URIToResource(String id) {
        super(id);
    }

    @Override
    public final void removeEntry(QueryProcessor provider) {
        provider.cache.remove(this);
    }

    public Object compute(ReadGraphImpl graph, final InternalProcedure<Integer> procedure) throws DatabaseException {
        computeForEach(graph, id, this, procedure);
        return getResult();
    }

    static void computeForEach(ReadGraphImpl graph, String id, final URIToResource entry, final InternalProcedure<Integer> procedure_) throws DatabaseException {

        InternalProcedure<Integer> procedure = entry != null ? entry : procedure_;

        if("http://".equals(id) || "http:/".equals(id)) {

            QueryProcessor processor = graph.processor;
            procedure.execute(graph, processor.getRootLibrary());

        } else {

            final String[] parts = URIStringUtils.splitURI(id);
            if (parts != null) {

                QueryCache.runnerURIToResource(graph, parts[0], entry, null, new InternalProcedure<Integer>() {

                    @Override
                    public void execute(ReadGraphImpl graph, Integer parentId) throws DatabaseException {

                        ObjectResourceIdMap<String> map = QueryCache.resultChildMap(graph, parentId, entry, null);
                        assert(map != null);
                        int result = map.getId(URIStringUtils.unescape(parts[1]));
                        if (result == 0) {
                            ResourceNotFoundException e = new ResourceNotFoundException(id);
                            procedure.exception(graph, e);
                        } else {
                            procedure.execute(graph, result);
                        }
                    }

                    @Override
                    public void exception(ReadGraphImpl graph, Throwable throwable) throws DatabaseException {
                        procedure.exception(graph, throwable);
                    }

                });

            } else {
                ResourceNotFoundException e = new ResourceNotFoundException(id);
                procedure.exception(graph, e);
            }

        }

        if(entry != null) entry.performFromCache(graph, procedure_);

    }

    public void addOrSet(Integer result) {

        assert(isPending());

        synchronized(this) {
            setResult(result);
            setReady();
        }

    }

    @Override
    public String toString() {
        return "URIToResource[" + id + "]";
    }

    @Override
    public Object performFromCache(ReadGraphImpl graph, InternalProcedure<Integer> procedure) throws DatabaseException {

        assert(isReady());

        if(handleException(graph, procedure)) return (Throwable)getResult();

        Integer result = (Integer)getResult();
        procedure.execute(graph, result);
        return result;

    }

    @Override
    public void recompute(ReadGraphImpl graph) throws DatabaseException {

        compute(graph, new InternalProcedure<Integer>() {

            @Override
            public void execute(ReadGraphImpl graph, Integer result) {
            }

            @Override
            public void exception(ReadGraphImpl graph, Throwable t) {
                if(DebugException.DEBUG) new DebugException(t).printStackTrace();
                throw new Error("Error in recompute.", t);
            }

        });

    }

    @Override
    public void execute(ReadGraphImpl graph, Integer result) throws DatabaseException {
        synchronized(this) {
            setResult(result);
            setReady();
        }
    }

    @Override
    public void exception(ReadGraphImpl graph, Throwable throwable) throws DatabaseException {
        except(throwable);
    }

}
