/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.common.auth;

import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.binding.error.BindingConstructionException;
import org.simantics.db.ReadGraph;
import org.simantics.db.RequestProcessor;
import org.simantics.db.Resource;
import org.simantics.db.authentication.UserAuthenticator;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.InvalidAuthenticationException;
import org.simantics.db.exception.InvalidUserException;
import org.simantics.db.request.Read;
import org.simantics.layer0.Layer0;
import org.simantics.user.UserResource;
import org.simantics.utils.bytes.Base64;

public final class UserAuthenticators {

    private static class Digest implements UserAuthenticator {
        private final String userName;
        private final String localDigest;
        private final String remoteDigest;

        Digest(String userName, String localDigest, String remoteDigest) {
            this.userName = userName;
            this.localDigest = localDigest;
            this.remoteDigest = remoteDigest;
        }

        @Override
        public Resource getUser(RequestProcessor processor) throws InvalidUserException, InvalidAuthenticationException {

            try {

                Resource user = processor.syncRequest(new Read<Resource>() {

                    @Override
                    public Resource perform(final ReadGraph graph) throws DatabaseException {

                        Resource userLibrary = graph.getResource("http://Users");
                        Resource consistsOf = graph.getResource(Layer0.URIs.ConsistsOf);
                        Resource hasName = graph.getResource(Layer0.URIs.HasLabel);
                        Resource hasPasswordHash = graph.getResource(UserResource.URIs.HasPasswordHash);

                        for(Resource r : graph.getObjects(userLibrary, consistsOf)) {

                        	try {

                        		String name = graph.getRelatedValue(r, hasName, Bindings.STRING);
	
	                            if(name.equals(userName)) {
	
	                                String graphDigest = graph.getRelatedValue(r, hasPasswordHash, Bindings.getBinding(String.class));
	
	                                if(graphDigest.equals(localDigest)) return r;
	                                else throw new InvalidAuthenticationException("Password was not valid for user '" + userName + "'");
	
	                            }
                        	} catch(BindingConstructionException e) {
                        		throw new DatabaseException(e);
                        	}
	                            
                        }

                        throw new InvalidUserException("User '" + userName + "' was not found.");

                    }

                });

                return user;

            } catch (InvalidAuthenticationException e) {

                throw e;

            } catch (InvalidUserException e) {

                throw e;

            } catch (DatabaseException e) {

                throw new InvalidAuthenticationException("Authentication failed, see cause for details.", e);

            }

        }

        @Override
        public String userName() {
            return userName;
        }

        @Override
        public String remoteDigest() {
            return remoteDigest;
        }
    }

    public static UserAuthenticator byNameAndPassword(String userName, String password) {
        try {
            MessageDigest digest = MessageDigest.getInstance("SHA-512");
            String localDigest = Base64.encode(digest.digest(password.getBytes("US-ASCII"))).replace("\n", "").replace("\r", "");
            String reversePassword = new StringBuilder(password).reverse().toString();
            String remoteDigest = Base64.encode(digest.digest(reversePassword.getBytes("US-ASCII"))).replace("\n", "").replace("\r", "");

            return new Digest(userName, localDigest, remoteDigest);
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        } catch (UnsupportedEncodingException e) {
            throw new Error( "The JVM is required to support UTF-8 and US-ASCII encodings.");
        }
    }

}
