/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.common;

import java.util.concurrent.TimeUnit;

import org.simantics.db.Session;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.management.ISessionContext;
import org.simantics.db.management.ISessionContextProvider;
import org.simantics.db.management.ISessionContextProviderSource;
import org.simantics.db.request.ReadInterface;
import org.simantics.db.request.WriteInterface;
import org.simantics.layer0.Layer0;
import org.simantics.operation.Layer0X;
import org.simantics.utils.threads.ThreadUtils;

/**
 * An internal facade for accessing basic Simantics platform services.
 * Usable without a graphical UI, i.e. in headless contexts.
 * 
 * Use org.simantics.Simantics instead where ever possible.
 */
public class SimanticsInternal {

    private static ISessionContextProviderSource providerSource = null;

    /**
     * Queue execution of a runnable. 
     * 
     * @param runnable
     */
    public static void async(Runnable runnable) {
        ThreadUtils.getBlockingWorkExecutor().execute(runnable);
    }

    public static void async(Runnable runnable, int delay, TimeUnit unit) {
        ThreadUtils.getTimer().schedule(runnable, delay, unit);
    }
    
    /**
     * Queue execution of a non-blocking runnable. Use this method with caution. 
     * A non-blocking runnable nevers locks anything, No Locks, No semaphores,
     * No Object.wait(), No synchronized() {} blocks.   
     * 
     * @param runnable a non-blocking runnable
     */
    public static void asyncNonblocking(Runnable runnable) {
        ThreadUtils.getNonBlockingWorkExecutor().execute(runnable);
    }

    /**
     * Schedule execution of a non-blocking runnable. Use this method with caution. 
     * A non-blocking runnable never locks anything, No Locks, No semaphores,
     * No Object,wait(), No synchronized() {} blocks.   
     * 
     * @param runnable a non-blocking runnable
     * @param initialDelay
     * @param period
     */
    public static void asyncNonblocking(Runnable runnable, int initialDelay, int period) {
        ThreadUtils.getNonBlockingWorkExecutor().scheduleAtFixedRate(runnable, initialDelay, period, TimeUnit.MILLISECONDS);
    }
    
    public static synchronized ISessionContext setSessionContext(ISessionContext ctx) {
        return getSessionContextProvider().setSessionContext(ctx);
    }

    public static void setSessionContextProviderSource(ISessionContextProviderSource source) {
        if (source == null)
            throw new IllegalArgumentException("null provider source");
        providerSource = source;
    }

    public static ISessionContextProviderSource getProviderSource() {
        if (providerSource == null)
            throw new IllegalStateException(
            "providerSource must be initialized by the application before using class Simantics");
        return providerSource;
    }

    public static ISessionContextProvider getSessionContextProvider() {
        return getProviderSource().getActive();
    }

    /**
     * Returns the database session context associated with the currently active
     * context. This method should be used to retrieve session contexts only
     * when the client is sure that the correct context is active.
     * 
     * @return the session context associated with the currently active context
     *         or <code>null</code> if the context has no session context
     */
    public static ISessionContext getSessionContext() {
        ISessionContextProvider provider = getSessionContextProvider();
        return provider != null ? provider.getSessionContext() : null;
    }

    /**
     * Returns the database Session bound to the currently active context.
     * 
     * <p>
     * The method always returns a non-null Session or produces an
     * IllegalStateException if a Session was not attainable.
     * </p>
     * 
     * @return the Session bound to the currently active workbench window
     * @throws IllegalStateException if no Session was available
     */
    public static Session getSession() {
        ISessionContext ctx = getSessionContext();
        if (ctx == null)
            throw new IllegalStateException("Session unavailable, no database session open");
        return ctx.getSession();
    }

    /**
     * Returns the database Session bound to the currently active context.
     * Differently from {@link #getSession()}, this method returns
     * <code>null</code> if there is no current Session available.
     * 
     * @see #getSession()
     * @return the Session bound to the currently active context or
     *         <code>null</code>
     */
    public static Session peekSession() {
        ISessionContext ctx = getSessionContext();
        return ctx == null ? null : ctx.peekSession();
    }

    public static Layer0 getLayer0() throws DatabaseException {
        return Layer0.getInstance(getSession());
    }

    public static Layer0X getLayer0X() throws DatabaseException {
        return Layer0X.getInstance(getSession());
    }

    
    public static <T> T sync(ReadInterface<T> r) throws DatabaseException {
    	return getSession().sync(r);
    }
    
    public static <T> T sync(WriteInterface<T> r) throws DatabaseException {
    	return getSession().sync(r);
    }
    
}
