package org.simantics.datatypes;

import java.io.File;
import java.io.PrintStream;
import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLDecoder;
import java.util.ArrayList;

public class SIGenerator {

	public static class Unit {
		
		public String name;
		public String shorthand;
		
		public Unit(String name, String shorthand) {
			this.name = name;
			this.shorthand = shorthand;
		}
		
		public Unit withPrefix(String namePrefix, String shorthandPrefix) {
			return new Unit(namePrefix+name, shorthandPrefix+shorthand);
		}
		
	}
	
	public static final Unit[] baseUnits = {
		new Unit("Metre", "m"),
		// Easier this way..
		new Unit("gram", "g"),
		new Unit("Second", "s"),
		new Unit("Ampere", "A"),
		new Unit("Kelvin", "K"),
		new Unit("Candela", "cd"),
		new Unit("Mole", "mol"),
	};
	
	public static final Unit[] baseUnitsPrefixed = prefix(baseUnits);
	
	public static final Unit[] derivedUnits = {
		new Unit("Hertz", "hz"),
		new Unit("Radian", "rad"),
		new Unit("Steradian", "sr"),
		new Unit("Newton", "N"),
		new Unit("Pascal", "Pa"),
		new Unit("Joule", "J"),
		new Unit("Watt", "W"),
		new Unit("Coulomb", "C"),
		new Unit("Volt", "V"),
		new Unit("Farad", "F"),
		new Unit("Ohm", "ohm"),
		new Unit("Siemens", "S"),
		new Unit("Weber", "Wb"),
		new Unit("Tesla", "T"),
		new Unit("Henry", "H"),
		new Unit("Celsius", "C"),
		new Unit("Lumen", "lm"),
		new Unit("Lux", "lx"),
		new Unit("Becquerel", "Bq"),
		new Unit("Gray", "Gy"),
		new Unit("Sievert", "Sv"),
		new Unit("Katal", "kat")
	};

	public static final Unit[] derivedUnitsPrefixed = prefix(derivedUnits);

	public static final Unit[] scalableCompoundUnits = {
		new Unit("gramPerSecond", "g/s"),
		new Unit("gramPerLitre", "g/l"),
		new Unit("gramPerCubicMetre", "g/m3"),
		new Unit("NewtonPerMetre", "N/m"),
		new Unit("JoulePerKilogram", "J/kg"),
		new Unit("MetrePerSecond", "m/s"),
		new Unit("MolePerLitre", "mol/l"),
		new Unit("MolePerKilogram", "mol/kg"),
	};

	public static final Unit[] scalableCompoundUnitsPrefixed = prefix(scalableCompoundUnits);

	public static final Unit[] compoundUnits = {
		new Unit("SquareMetre", "m2"),
		new Unit("CubicMetre", "m3"),
	};

	public static final Unit[] specials = {
		new Unit("MassPercentage", "w-%"),
		new Unit("Percentage", "%"),
		new Unit("Degree", "deg"),
		new Unit("Minute", "min"),
		new Unit("Hour", "h"),
		new Unit("Litre", "l"),
	};
	
	static Unit[] prefix(Unit[] units) {
		ArrayList<Unit> result = new ArrayList<Unit>();
		for(Unit unit : units) {
			result.add(unit.withPrefix("Tera", "T"));
			result.add(unit.withPrefix("Giga", "G"));
			result.add(unit.withPrefix("Mega", "M"));
			result.add(unit.withPrefix("Kilo", "k"));
			result.add(unit.withPrefix("Hecto", "h"));
			result.add(unit.withPrefix("", ""));
			result.add(unit.withPrefix("Centi", "c"));
			result.add(unit.withPrefix("Milli", "m"));
			result.add(unit.withPrefix("Micro", "u"));
		}
		return result.toArray(new Unit[result.size()]);
	}
	
	static void print(PrintStream out, Unit unit) {
		out.println("SI." + unit.name + " <T SI.Unit");
		out.println("SI." + unit.name + ".Double <T L0.Double");
		out.println("  @L0.assert L0.HasDataType $Double(unit=\"" + unit.shorthand + "\")");
		out.println("SI." + unit.name + ".Float <T L0.Float");
		out.println("  @L0.assert L0.HasDataType $Float(unit=\"" + unit.shorthand + "\")");
		out.println("");
	}
	
	static void create(File path) {
		
		try {

			PrintStream out = new PrintStream(path + "/graph/SI.pgraph");
			out.println("L0 = <http://www.simantics.org/Layer0-1.1>");
			out.println("DATA = <http://www.simantics.org/Datatypes-1.1>");
			out.println("SI = DATA.SI : L0.Library");
			out.println("    L0.HasResourceClass \"org.simantics.datatypes.SIResource\"");
			out.println("");
			out.println("SI.Unit <T L0.Entity");
			out.println("");
			out.println("// SI base units");
			out.println("");
			
			for(Unit unit : baseUnitsPrefixed) {
				print(out, unit);
			}

			out.println("// SI derived units");
			out.println("");

			for(Unit unit : derivedUnitsPrefixed) {
				print(out, unit);
			}

			out.println("// SI scalable compound units");
			out.println("");

			for(Unit unit : scalableCompoundUnitsPrefixed) {
				print(out, unit);
			}
			
			out.println("// SI compound units");
			out.println("");

			for(Unit unit : compoundUnits) {
				print(out, unit);
			}

			out.println("// Special units");
			out.println("");

			for(Unit unit : specials) {
				print(out, unit);
			}
			
		} catch (Exception e) {
			
		}
		
	}
	
	public static void main(String[] args) {
		
        URL classLocation = SIGenerator.class.getResource(".");
        if (classLocation != null) {
            if (classLocation.getProtocol().equals("file")) {
                try {
                    URL resource = new URL(classLocation, "../../../..");
                    File path = new File(URLDecoder.decode(resource.getPath(), "UTF-8"));
                    System.err.println("path=" + path.getAbsolutePath());
                    create(path);
                } catch (MalformedURLException e) {
                    e.printStackTrace();
                } catch (UnsupportedEncodingException e) {
                    e.printStackTrace();
                }
            }
        }
		
	}
	
}
