/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.swt;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.viewers.IPostSelectionProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.ui.IWorkbenchSite;
import org.simantics.Simantics;
import org.simantics.browsing.ui.GraphExplorer;
import org.simantics.browsing.ui.swt.stubs.BrowsingResource;
import org.simantics.browsing.ui.swt.widgets.GraphExplorerComposite;
import org.simantics.browsing.ui.swt.widgets.ModelBrowser;
import org.simantics.browsing.ui.swt.widgets.impl.WidgetSupport;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.management.ISessionContext;
import org.simantics.db.request.Read;

public class ModelledGraphExplorerComposite implements ModelledControl {

	final private Resource configuration;
	
	public ModelledGraphExplorerComposite(Resource configuration) {
		this.configuration = configuration;
	}
	
	@Override
	public Composite create(Composite parent, IWorkbenchSite site, ISessionContext context, final WidgetSupport support) throws DatabaseException {

		int style = Simantics.getSession().syncRequest(new Read<Integer>() {

			@Override
			public Integer perform(ReadGraph graph) throws DatabaseException {
				int style = SWT.BORDER | SWT.FULL_SELECTION | SWT.MULTI;
				BrowsingResource br = BrowsingResource.getInstance(graph);
				Boolean check = graph.getPossibleRelatedValue(configuration, br.GraphExplorerComposite_Check, Bindings.BOOLEAN);
				if(check != null && check) style |= SWT.CHECK;
				return style;
				
			}
			
		});
		
		Map<String, Object> args = Simantics.getSession().syncRequest(new Read<Map<String, Object>>() {

			@Override
			public Map<String, Object> perform(ReadGraph graph) throws DatabaseException {
				
		        Map<String, Object> args = new HashMap<String, Object>();
		        args.put("displaySelectors", Boolean.FALSE);
		        args.put("displayFilter", Boolean.FALSE);
		        
				BrowsingResource br = BrowsingResource.getInstance(graph);
				Boolean displayFilter = graph.getPossibleRelatedValue(configuration, br.GraphExplorerComposite_DisplayFilter, Bindings.BOOLEAN);
				if(displayFilter != null) args.put("displayFilter", displayFilter);
		        
		        return args;
		        
			}
			
		});

		String browseContextURI = Simantics.getSession().syncRequest(new Read<String>() {

			@Override
			public String perform(ReadGraph graph) throws DatabaseException {
				BrowsingResource br = BrowsingResource.getInstance(graph);
				Resource browseContext = graph.getSingleObject(configuration, br.GraphExplorerComposite_BrowseContext);
				return graph.getURI(browseContext);
			}
			
		});
		
        final GraphExplorerComposite explorer = new ModelBrowser(Collections.singleton(browseContextURI), args, site, parent, support, style) {
            @Override
            protected void addListeners(GraphExplorer explorer, IMenuManager menuManager) {
                // No mouse/key listeners in this explorer.
                // Prevents unwanted contributions from interfering
                // with how the control works.
            }
        };
        
        GridData gridData = Simantics.getSession().syncRequest(new Read<GridData>() {

            @Override
            public GridData perform(ReadGraph graph) throws DatabaseException {
                BrowsingResource br = BrowsingResource.getInstance(graph);
                GridData data = new GridData(SWT.FILL, SWT.FILL, true, true);                
                Integer span = graph.getPossibleRelatedValue(configuration, br.Control_HorizontalSpan, Bindings.INTEGER);
                if(span != null) {
                	data.horizontalSpan = span;
                }
                Integer preferredWidth = graph.getPossibleRelatedValue(configuration, br.Control_PreferredWidth, Bindings.INTEGER);
                data.widthHint = preferredWidth != null ? preferredWidth : SWT.DEFAULT;
                Integer preferredHeight = graph.getPossibleRelatedValue(configuration, br.Control_PreferredHeight, Bindings.INTEGER);
                data.heightHint = preferredHeight != null ? preferredHeight : SWT.DEFAULT;
                return data;
            }

        });
        
        explorer.setLayoutData(gridData);
        
//        explorer.setBrowseContexts(Collections.singleton(browseContextURI));
        
        explorer.setInputSource(new InputSourceImpl<Object>() {

			@Override
			public Object get(ISessionContext ctx, Object selection) {
				new Exception(""+selection).printStackTrace();
				return selection;
			}
		});
        
		Listener selectionListener = Simantics.getSession().syncRequest(new Read<Listener>() {

			@Override
			public Listener perform(ReadGraph graph) throws DatabaseException {
				BrowsingResource br = BrowsingResource.getInstance(graph);
				Resource listener = graph.getPossibleObject(configuration, br.GraphExplorerComposite_SelectionListener);
				if(listener == null) return null;
				return graph.adapt(listener, Listener.class);
			}
			
		});
		
		if(selectionListener != null) explorer.addListenerToControl(SWT.Selection, selectionListener);
		
		final String selectionParameter = Simantics.getSession().syncRequest(new Read<String>() {

			@Override
			public String perform(ReadGraph graph) throws DatabaseException {
				BrowsingResource br = BrowsingResource.getInstance(graph);
				Resource parameter = graph.getPossibleObject(configuration, br.GraphExplorerComposite_SelectionParameter);
				if(parameter == null) return null;
				return graph.getPossibleURI(parameter);
			}
			
		});
		
		
		if(selectionParameter != null) {
			IPostSelectionProvider selectionProvider = (IPostSelectionProvider)explorer.getExplorer().getAdapter(IPostSelectionProvider.class);
			selectionProvider.addPostSelectionChangedListener(new ISelectionChangedListener() {
				
				@Override
				public void selectionChanged(SelectionChangedEvent event) {
					ISelection selection = event.getSelection();
					support.setParameter(selectionParameter, selection);
				}
				
			});
		}
		
        explorer.finish();
        
        return explorer;
		
	}
	
}
