/*******************************************************************************
 * Copyright (c) 2007, 2012 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.swt;

import java.lang.reflect.Method;
import java.util.function.BiFunction;

import org.eclipse.core.runtime.Platform;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.services.IServiceLocator;
import org.osgi.framework.Bundle;
import org.simantics.Simantics;
import org.simantics.browsing.ui.BuiltinKeys;
import org.simantics.browsing.ui.GraphExplorer;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.SelectionDataResolver;
import org.simantics.browsing.ui.SelectionFilter;
import org.simantics.browsing.ui.common.AdaptableHintContext;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.PossibleTypedParent;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.common.utils.Logger;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.SelectionHints;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.simulation.ontology.SimulationResource;
import org.simantics.utils.datastructures.hints.IHintContext;
import org.simantics.utils.ui.ExceptionUtils;

/**
 * @author Tuukka Lehtonen
 */
public class GraphExplorerFactory {

    private int                     maxChildrenShown      = GraphExplorerImpl.DEFAULT_MAX_CHILDREN;

    private SelectionDataResolver  selectionDataResolver;
    private SelectionFilter        selectionFilter;

    private IServiceLocator        serviceLocator;

    private BiFunction<GraphExplorer, Object[], Object[]> selectionTransformation = new BiFunction<GraphExplorer, Object[], Object[]>() {

    	private Resource getModel(final Object object) {
    		if(object instanceof NodeContext) {
    			NodeContext context = (NodeContext)object;
    			Object input = context.getConstant(BuiltinKeys.INPUT);
    			if(input instanceof Resource) {
    				final Resource inputResource = (Resource)input;
    	    		try {
            			return Simantics.getSession().sync(new UniqueRead<Resource>() {

            				@Override
            				public Resource perform(ReadGraph graph) throws DatabaseException {
            					SimulationResource SIMU = SimulationResource.getInstance(graph);
            					return graph.sync(new PossibleTypedParent(inputResource, SIMU.Model));
            				}

            			});
    	    		} catch (DatabaseException e) {
    	    			Logger.defaultLogError(e);
    	    		}
    			} else if (input instanceof Variable) {
    				final Variable inputVariable = (Variable)input;
    	    		try {
            			return Simantics.getSession().sync(new UniqueRead<Resource>() {

            				@Override
            				public Resource perform(ReadGraph graph) throws DatabaseException {
            					return Variables.getModel(graph, inputVariable);
            				}

            			});
    	    		} catch (DatabaseException e) {
    	    			Logger.defaultLogError(e);
    	    		}
    			}
    		}
    		return null;
    	}
    	
        @Override
        public Object[] apply(GraphExplorer explorer, Object[] objects) {
            Object[] result = new Object[objects.length];
            for (int i = 0; i < objects.length; i++) {
                IHintContext context = new AdaptableHintContext(SelectionHints.KEY_MAIN);
                context.setHint(SelectionHints.KEY_MAIN, objects[i]);
                Resource model = getModel(objects[i]);
                if(model != null)
                	context.setHint(SelectionHints.KEY_MODEL, model);
                result[i] = context;
            }
            return result;
        }

    };

    public static GraphExplorerFactory getInstance() {
        return new GraphExplorerFactory();
    }

    /**
     * @param n
     * @return this instance
     */
    public GraphExplorerFactory maxChildrenShown(int n) {
        this.maxChildrenShown = n;
        return this;
    }

    public GraphExplorerFactory selectionTransformation(BiFunction<GraphExplorer, Object[], Object[]> transformation) {
        this.selectionTransformation = transformation;
        return this;
    }

    /**
     * @param r
     * @return this instance
     */
    public GraphExplorerFactory selectionDataResolver(SelectionDataResolver r) {
        this.selectionDataResolver = r;
        return this;
    }

    public GraphExplorerFactory setSelectionFilter(SelectionFilter filter) {
        this.selectionFilter = filter;
        return this;
    }

    public GraphExplorerFactory setServiceLocator(IServiceLocator serviceLocator) {
        this.serviceLocator = serviceLocator;
        return this;
    }

    public GraphExplorer create(Composite parent) {
        return create(parent, 0);
    }

    /**
     * @param site
     * @param parent
     * @param style SWT style hints for the explorer tree control to create
     * @return
     */
    public GraphExplorer create(Composite parent, int style) {
        return createWithStyle(parent, style | SWT.VIRTUAL);
        //return createWithStyle(parent, style);
    }

    private static class GraphExplorerWithMaxChildren extends GraphExplorerImpl {
//    	private int maxChildrenShown;
    	GraphExplorerWithMaxChildren(Composite parent, int style, int maxChildrenShown) {
    		super(parent, style);
    		setMaxChildren(maxChildrenShown);
//    		this.maxChildrenShown = maxChildrenShown;
    	}
//    	@Override
//    	public int getMaxChildren() {
//    		return maxChildrenShown;
//    	}
    }
    
    public GraphExplorer createWithStyle(Composite parent, int style) {
        GraphExplorerImpl explorer = new GraphExplorerWithMaxChildren(parent, style, maxChildrenShown);
        explorer.setSelectionDataResolver(selectionDataResolver);
        explorer.setSelectionFilter(selectionFilter);
        explorer.setSelectionTransformation(selectionTransformation);
        explorer.setServiceLocator(serviceLocator);
        return explorer;
    }
    
    public GraphExplorer create2(Composite parent, int style) {
        GraphExplorerImpl2 explorer = new GraphExplorerImpl2(parent, style);
        explorer.setSelectionDataResolver(selectionDataResolver);
        explorer.setSelectionFilter(selectionFilter);
        explorer.setSelectionTransformation(selectionTransformation);
        explorer.setServiceLocator(serviceLocator);
        return explorer;
    }
    
    public GraphExplorer create3(Composite parent, int style) {
        //GraphExplorerImpl2 explorer = new GraphExplorerImpl2(parent, style);
    	try {
	    	Bundle bundle = Platform.getBundle("org.simantics.browsing.ui.nattable");
	    	@SuppressWarnings("unchecked")
			Class<GraphExplorer> clazz = (Class<GraphExplorer>)bundle.loadClass("org.simantics.browsing.ui.nattable.NatTableGraphExplorer");
	    	//Class<GraphExplorer> clazz = (Class<GraphExplorer>)bundle.getClass().getClassLoader().loadClass("org.simantics.browsing.ui.nattable.NatTableGraphExplorer");
	    	GraphExplorer explorer = clazz.getConstructor(Composite.class, int.class).newInstance(parent,style);
	        explorer.setSelectionDataResolver(selectionDataResolver);
	        explorer.setSelectionFilter(selectionFilter);
	        explorer.setSelectionTransformation(selectionTransformation);
	        Method m = clazz.getMethod("setServiceLocator", IServiceLocator.class);
	        m.invoke(explorer, serviceLocator);
	        //explorer.setServiceLocator(serviceLocator);
	        return explorer;
    	} catch (Throwable t) {
    		ExceptionUtils.logAndShowError(t);
    		return null;
    	}
    }

//    void hookActions(IWorkbenchSite site) {
//        IActionBars actionBars = null;
//
//        if (site instanceof IEditorSite)
//            actionBars = ((IEditorSite) site).getActionBars();
//        if (site instanceof IViewSite)
//            actionBars = ((IViewSite) site).getActionBars();
//        if (site instanceof IPageSite)
//            actionBars = ((IPageSite) site).getActionBars();
//
//        if (actionBars != null) {
//            actionBars.setGlobalActionHandler(ActionFactory.SELECT_ALL.getId(), new Action() {
//                @Override
//                public void run() {
//                    System.out.println("SELECT_ALL");
//                }
//            });
//            actionBars.setGlobalActionHandler(ActionFactory.FIND.getId(), new Action() {
//                @Override
//                public void run() {
//                    System.out.println("FIND");
//                }
//            });
//            actionBars.setGlobalActionHandler(ActionFactory.UNDO.getId(), new Action() {
//                @Override
//                public void run() {
//                    System.out.println("UNDO");
//                    // SimanticsUI.undo();
//                }
//            });
//            actionBars.setGlobalActionHandler(ActionFactory.REDO.getId(), new Action() {
//                @Override
//                public void run() {
//                    System.out.println("REDO");
//                    // SimanticsUI.redo();
//                }
//            });
//            actionBars.updateActionBars();
//        }
//    }

}
