/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.swt;

import gnu.trove.map.hash.THashMap;
import gnu.trove.set.hash.THashSet;

import java.util.Collection;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Tree;
import org.simantics.browsing.ui.BuiltinKeys;
import org.simantics.browsing.ui.GraphExplorer;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.NodeContext.PrimitiveQueryKey;
import org.simantics.browsing.ui.PrimitiveQueryUpdater;
import org.simantics.browsing.ui.common.processors.AbstractPrimitiveQueryProcessor;
import org.simantics.browsing.ui.common.processors.IsExpandedProcessor;
import org.simantics.browsing.ui.common.processors.ProcessorLifecycle;

/**
 * @author Tuukka Lehtonen
 */
public class DefaultIsExpandedProcessor extends AbstractPrimitiveQueryProcessor<Boolean> implements
IsExpandedProcessor, ProcessorLifecycle {

    /**
     * The set of currently expanded node contexts.
     */
    private final Set<NodeContext>                        expanded        = new THashSet<NodeContext>();
    private final Map<NodeContext, PrimitiveQueryUpdater> expandedQueries = new THashMap<NodeContext, PrimitiveQueryUpdater>();

    private Tree tree;

    public DefaultIsExpandedProcessor() {
    }

    @Override
    public Object getIdentifier() {
        return BuiltinKeys.IS_EXPANDED;
    }

    @Override
    public String toString() {
        return "IsExpandedProcessor";
    }

    @Override
    public Boolean query(PrimitiveQueryUpdater updater, NodeContext context, PrimitiveQueryKey<Boolean> key) {
        boolean isExpanded = expanded.contains(context);
        //System.out.println("isExpanded(" + updater + ", " + context + "): " + isExpanded);
        expandedQueries.put(context, updater);
        return Boolean.valueOf(isExpanded);
    }

    @Override
    public Collection<NodeContext> getExpanded() {
        return new HashSet<NodeContext>(expanded);
    }

    @Override
    public boolean getExpanded(NodeContext context) {
        return this.expanded.contains(context);
    }

    @Override
    public boolean setExpanded(NodeContext context, boolean expanded) {
        return _setExpanded(context, expanded);
    }

    @Override
    public boolean replaceExpanded(NodeContext context, boolean expanded) {
        return nodeStatusChanged(context, expanded);
    }

    private boolean _setExpanded(NodeContext context, boolean expanded) {
        if (expanded) {
            return this.expanded.add(context);
        } else {
            return this.expanded.remove(context);
        }
    }

    Listener treeListener = new Listener() {
        @Override
        public void handleEvent(Event event) {
            NodeContext context = (NodeContext) event.item.getData();
            switch (event.type) {
                case SWT.Expand:
                    nodeStatusChanged(context, true);
                    break;
                case SWT.Collapse:
                    nodeStatusChanged(context, false);
                    break;
            }
        }
    };

    protected boolean nodeStatusChanged(NodeContext context, boolean expanded) {
        boolean result = _setExpanded(context, expanded);
        PrimitiveQueryUpdater updater = expandedQueries.get(context);
        if (updater != null)
            updater.scheduleReplace(context, BuiltinKeys.IS_EXPANDED, expanded);
        return result;
    }

    @Override
    public void attached(GraphExplorer explorer) {
        Object control = explorer.getControl();
        if (control instanceof Tree) {
            this.tree = (Tree) control;
            tree.addListener(SWT.Expand, treeListener);
            tree.addListener(SWT.Collapse, treeListener);
        } else {
            System.out.println("WARNING: " + getClass().getSimpleName() + " attached to unsupported control: " + control);
        }
    }

    @Override
    public void clear() {
        expanded.clear();
        expandedQueries.clear();
    }

    @Override
    public void detached(GraphExplorer explorer) {
        clear();
        if (tree != null) {
            tree.removeListener(SWT.Expand, treeListener);
            tree.removeListener(SWT.Collapse, treeListener);
            tree = null;
        }
    }

}