/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.graph.impl.contribution;

import java.util.Collection;

import org.simantics.browsing.ui.BuiltinKeys;
import org.simantics.browsing.ui.DataSource;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.PrimitiveQueryUpdater;
import org.simantics.browsing.ui.Tester;
import org.simantics.browsing.ui.content.ViewpointContribution;
import org.simantics.browsing.ui.graph.impl.request.ResourceQuery;
import org.simantics.db.AsyncReadGraph;
import org.simantics.db.ReadGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.procedure.Listener;
import org.simantics.db.procedure.Procedure;
import org.simantics.utils.ui.ErrorLogger;

/**
 * Implement {@link #children(AsyncReadGraph)} and {@link #hasChildren(AsyncReadGraph)}.
 * 
 * @author Tuukka Lehtonen
 */
abstract public class FinalViewpointContributionImpl extends ContributionStub implements GraphContribution {

    final private ResourceQuery<Collection<NodeContext>> childQuery;
    final private Procedure<Collection<NodeContext>>     childProcedure;

    final protected PrimitiveQueryUpdater       updater;
    final private NodeContext                context;
    final private BuiltinKeys.ViewpointContributionKey    key;

    /**
     * This identity is used to give the back-end graph requests a
     * <em>properly unique</em> identity that so that the graph back-end caching
     * and graph explorer node context caching work together properly.
     * 
     * Consider having two graph explorer instances that have the same
     * configuration (same evaluators) and are showing the same resource from
     * the graph database. In this case the requests are actually meant to have
     * an identical identity and performing the graph request will simply bind a
     * new listener for the one and same request.
     * 
     * @return an additional identity for graph back-end requests to make them
     *         properly unique
     */
    public Object getIdentity() {
        return key;
    }

    public FinalViewpointContributionImpl(final PrimitiveQueryUpdater updater, NodeContext context, BuiltinKeys.ViewpointContributionKey key) {

        assert updater != null;
        assert context != null;
        assert key != null;

        this.updater = updater;
        this.context = context;
        this.key = key;

        this.childQuery = new ResourceQuery<Collection<NodeContext>>(getIdentity(), context) {

            @Override
            public Collection<NodeContext> perform(ReadGraph graph) throws DatabaseException {
                try {
                    // Make sure that null is not returned.
                    Collection<NodeContext> result = children(graph, context);
                    if (result == null)
                        throw new NullPointerException("LazyContributionImpl.children is not allowed to return null, but " + FinalViewpointContributionImpl.this.getClass() + " just did it");
                    return result;
                } catch (DatabaseException e) {
                    throw e;
                } catch (Throwable t) {
                    ErrorLogger.defaultLogError("LazyContributionImpl.childQuery produced unexpected exception.", t);
                    return ViewpointContribution.NO_CONTRIBUTION;
                }
            }

            @Override
            public String toString() {
                return "LazyContributionImpl[" + System.identityHashCode(FinalViewpointContributionImpl.this) + "].childQuery";
            }

        };

        childProcedure = createProcedure();
        
    }
    
    protected Procedure<Collection<NodeContext>> createProcedure() {

    	return new Procedure<Collection<NodeContext>>() {

            @Override
            public void execute(Collection<NodeContext> result) {
                replaceChildrenResult(result);
            }

            public void exception(Throwable t) {
                ErrorLogger.defaultLogError("LazyContributionImpl.childQuery failed, see exception for details.", t);
            }

            @Override
            public String toString() {
                return "LazyContributionImpl[" + System.identityHashCode(FinalViewpointContributionImpl.this) + "].childProcedure";
            }

        };
    	
    }

    public NodeContext getContext() {
        return context;
    }

    @Override
    public Collection<NodeContext> getContribution() {

        //System.out.println("LazyViewpoint2@" + System.identityHashCode(this) + " getChildren() = " + children.length);

        if (children == org.simantics.browsing.ui.content.ViewpointContribution.PENDING_CONTRIBUTION) {
            DataSource<AsyncReadGraph> source = updater.getDataSource(AsyncReadGraph.class);
            if (source != null) {
                source.schedule(graph -> {
                    if(childProcedure instanceof Listener<?>)
                        graph.asyncRequest(childQuery, (Listener<Collection<NodeContext>>)childProcedure);
                    else 
                        graph.asyncRequest(childQuery, childProcedure);
                });
            }
        }

        //System.out.println("LazyViewpoint.getChildren returns " + children);

        return children;

    }

    protected void replaceChildrenResult(Collection<NodeContext> result) {
        setChildren(updater, result);
        updater.scheduleReplace(context, key, this);
    }

    /**
     * @param <T>
     * @param clazz
     * @return input of the specified class
     * @throws ClassCastException if the input class does not match the
     *         specified class
     * @throws NullPointerException if the input is null
     */
    @SuppressWarnings("unchecked")
    protected <T> T getInput(Class<T> clazz) throws ClassCastException {
        Object o = context.getConstant(BuiltinKeys.INPUT);
        if (o == null)
            throw new NullPointerException("null input");
//        return clazz.cast(o);
        return (T) o;
    }

    /**
     * @param <T>
     * @param clazz
     * @return <code>null</code> if input is <code>null</code> or if the class does not match
     */
    @SuppressWarnings("unchecked")
    protected <T> T tryGetInput(Class<T> clazz) {
        Object o = context.getConstant(BuiltinKeys.INPUT);
        if (o != null && clazz.isInstance(o))
//            return clazz.cast(o);
            return (T) o;
        return null;
    }

    @Override
    public Tester getNodeContextTester() {
        return null;
    }

}
