/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.common.modifiers;

import java.util.Arrays;
import java.util.List;

/**
 * @author Tuukka Lehtonen
 * 
 * @param <T>
 */
public class Enumeration<T> {

    private final List<EnumeratedValue<T>> values;
    private final transient int hash;

    public static <T> Enumeration<T> make(List<EnumeratedValue<T>> values) {
        return new Enumeration<T>(values);
    }

    @SafeVarargs
    public static <T> Enumeration<T> make(EnumeratedValue<T>... values) {
        return new Enumeration<T>(Arrays.asList(values));
    }

    public Enumeration(List<EnumeratedValue<T>> values) {
        assert values != null;
        this.values = values;
        hash = values.hashCode();
    }

    public List<EnumeratedValue<T>> values() {
        return values;
    }

    public boolean contains(EnumeratedValue<T> value) {
        return values.contains(value);
    }

    public int indexOf(EnumeratedValue<T> value) {
        return values.indexOf(value);
    }

    public int size() {
        return values.size();
    }

    /**
     * @param object
     * @return the enumerated value matching the specified object
     * @throws IllegalArgumentException if matching EnumeratedValue not found
     * @throws NullPointerException if null object is specified
     */
    public EnumeratedValue<T> get(T object) {
        if (object == null)
            throw new NullPointerException("null object");
        for (EnumeratedValue<T> value : values)
            if (object.equals(value.getObject()))
                return value;
        throw new IllegalArgumentException("object '" + object + "' not found among enumeration " + values);
    }

    /**
     * Find an EnumeratedValue<T> matching the specified name.
     * 
     * @param name
     * @return <code>null</code> if corresponding EnumeratedValue not found
     */
    public EnumeratedValue<T> findByName(String name) {
        if (name == null)
            return null;
        for (EnumeratedValue<T> value : values)
            if (name.equals(value.getName()))
                return value;
        return null;
    }

    /**
     * Find an EnumeratedValue<T> matching the specified name.
     * 
     * @param name
     * @return <code>null</code> if corresponding EnumeratedValue not found
     */
    public EnumeratedValue<T> findByNameStartsWith(String name) {
        if (name == null)
            return null;
        for (EnumeratedValue<T> value : values)
            if (value.getName().startsWith(name))
                return value;
        return null;
    }

    /**
     * Find an EnumeratedValue<T> matching the specified object.
     * 
     * @param object
     * @return <code>null</code> if corresponding EnumeratedValue not found
     */
    public EnumeratedValue<T> find(T object) {
        if (object == null)
            return null;
        for (EnumeratedValue<T> value : values)
            if (object.equals(value.getObject()))
                return value;
        return null;
    }

    @Override
    public String toString() {
        return values.toString();
    }

    @Override
    public int hashCode() {
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        Enumeration<?> other = (Enumeration<?>) obj;
        return values.equals(other.values);
    }

}
