/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.common.modifiers;

import java.util.ArrayList;
import java.util.List;

import org.simantics.browsing.ui.content.Labeler.EnumerationModifier;

/**
 * A trivial EnumerationModifier base implementation
 * 
 * @author Tuukka Lehtonen
 */
public abstract class AbstractEnumerationModifier<T> implements EnumerationModifier {

    private final Enumeration<T>     enumeration;
    private final EnumeratedValue<T> value;
    private final List<String>       values;

    public AbstractEnumerationModifier(Enumeration<T> enumeration, EnumeratedValue<T> value) {
        assert enumeration != null;
        assert enumeration.size() > 0;

        this.enumeration = enumeration;
        this.value = value;

        this.values = new ArrayList<String>(enumeration.size());
        for (EnumeratedValue<T> v : enumeration.values())
            values.add(v.getName());
    }

    @Override
    public final List<String> getValues() {
        return values;
    }

    @Override
    public final String getValue() {
        if (value != null)
            return value.getName();
        return enumeration.values().get(0).getName();
    }

    @Override
    public final String isValid(String label) {
        if (!values.contains(label))
            return "Value '" + label + "' is not among the enumerated values " + values;
        return null;
    }

    @Override
    public final void modify(String label) {
        int index = values.indexOf(label);
        if (index == -1)
            throw new IllegalArgumentException("Cannot modify enumeration with value '" + label + "', not among the enumerated values " + values);
        modifyWithValue(value, enumeration.values().get(index));
    }

    protected void modifyWithValue(EnumeratedValue<T> oldEnumValue, EnumeratedValue<T> enumValue) {
        T oldObject = oldEnumValue != null ? oldEnumValue.getObject() : null;
        T newObject = enumValue != null ? enumValue.getObject() : null;
        modifyWithObject(oldObject, newObject);
    }

    /**
     * At least implement this if you don't override
     * {@link #modifyWithValue(EnumeratedValue, EnumeratedValue)}.
     * 
     * @param oldEnumObject
     * @param enumObject
     */
    protected void modifyWithObject(T oldEnumObject, T enumObject) {
    }

};
