/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.application.arguments;

import java.util.ListIterator;

/**
 * @author Tuukka Lehtonen
 *
 * @param <T>
 */
public interface IArgumentFactory<T> {

    String getArgument();

    IArgument<T> parse(String arg, ListIterator<String> tail);
    
    abstract class ArgumentFactory<V> implements IArgumentFactory<V> {
        String arg;
        protected IArgumentValidator<V> validator;
        public ArgumentFactory(String arg) {
            this(arg, null);
        }
        public ArgumentFactory(String arg, IArgumentValidator<V> validator) {
            this.arg = arg;
            this.validator = validator;
        }
        public String getArgument() {
            return arg;
        }
        protected IArgumentValidator<V> getValidator() {
            return validator;
        }
    }
    
    abstract class Argument<T> implements IArgument<T> {
        String key;

        public Argument(String key) {
            assert key != null;
            this.key = key;
        }

        @Override
        public String getArgument() {
            return key;
        }

        public abstract T getValue();

        @Override
        public String toString() {
            return "(" + getValue().getClass().getSimpleName() + ") " + getValue().toString();
        }
    }
    
    class BooleanArgument extends Argument<Boolean> {
        Boolean value;
        public BooleanArgument(String key, Boolean value) {
            super(key);
            this.value = value;
        }
        public Boolean getValue() {
            return value;
        }
    };
    
    class StringArgument extends Argument<String> {
        String value;
        public StringArgument(String key, String value) {
            super(key);
            this.value = value;
        }
        public String getValue() {
            return value;
        }
    };
    
    class IntegerArgument extends Argument<Integer> {
        Integer value;
        public IntegerArgument(String key, Integer value) {
            super(key);
            this.value = value;
        }
        public Integer getValue() {
            return value;
        }
    };

    class LongArgument extends Argument<Long> {
        Long value;
        public LongArgument(String key, Long value) {
            super(key);
            this.value = value;
        }
        public Long getValue() {
            return value;
        }
    };

    
    class StringArgumentFactory extends ArgumentFactory<String> {
        public StringArgumentFactory(String key) {
            super(key);
        }
        public StringArgumentFactory(String key, IArgumentValidator<String> validator) {
            super(key, validator);
        }
        @Override
        public IArgument<String> parse(String key, ListIterator<String> tail) {
            if (getArgument().equalsIgnoreCase(key)) {
                if (tail.hasNext()) {
                    String next = tail.next();
                    IArgumentValidator<String> validator = getValidator();
                    if(validator != null) {
	                    String valid = validator.isValid(next);
	                    if (valid != null) {
	                        tail.previous();
	                        throw new IllegalArgumentException(valid);
	                    }
                    }
                    return new StringArgument(getArgument(), next);
                }
            }
            return null;
        }
    }
    
    class IntegerArgumentFactory extends ArgumentFactory<Integer> {
        private Integer lowLimit;
        private Integer highLimit;
        public IntegerArgumentFactory(String key) {
            this(key, null, null);
        }
        public IntegerArgumentFactory(String key, Integer lowLimit, Integer highLimit) {
            super(key);
            this.lowLimit = lowLimit;
            this.highLimit = highLimit;
        }
        @Override
        public IArgument<Integer> parse(String key, ListIterator<String> tail) {
            if (getArgument().equalsIgnoreCase(key)) {
                if (tail.hasNext()) {
                    try {
                        Integer l = Integer.valueOf(tail.next());
                        if (lowLimit != null && lowLimit > l)
                            throw new IllegalArgumentException("value is less than low limit, " + l + " < " + lowLimit);
                        if (highLimit != null && highLimit < l)
                            throw new IllegalArgumentException("value is greater than high limit, " + l + " > " + highLimit);
                        return new IntegerArgument(getArgument(), l);
                    } catch (NumberFormatException e) {
                        throw new IllegalArgumentException(e);
                    }
                }
            }
            return null;
        }
    }
    
    class LongArgumentFactory extends ArgumentFactory<Long> {
        public LongArgumentFactory(String key) {
            super(key);
        }
        @Override
        public IArgument<Long> parse(String key, ListIterator<String> tail) {
            if (getArgument().equalsIgnoreCase(key)) {
                if (tail.hasNext()) {
                    try {
                        Long l = Long.valueOf(tail.next());
                        return new LongArgument(getArgument(), l);
                    } catch (NumberFormatException e) {
                        throw new IllegalArgumentException(e);
                    }
                }
            }
            return null;
        }
    }
    
    class NoValueArgumentFactory extends ArgumentFactory<Boolean> {
        public NoValueArgumentFactory(String key) {
            super(key);
        }
        @Override
        public IArgument<Boolean> parse(String key, ListIterator<String> tail) {
            if (getArgument().equalsIgnoreCase(key)) {
                return new BooleanArgument(getArgument(), Boolean.TRUE);
            }
            return null;
        }
    }

}
