/*******************************************************************************
 * Copyright (c) 2007, 2018 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl.query;

import java.util.Collection;

import org.simantics.databoard.Bindings;
import org.simantics.db.AsyncReadGraph;
import org.simantics.db.DevelopmentKeys;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.impl.BlockingAsyncProcedure;
import org.simantics.db.impl.graph.ReadGraphImpl;
import org.simantics.db.impl.query.QueryProcessor.SessionTask;
import org.simantics.db.procedure.AsyncProcedure;
import org.simantics.db.request.AsyncRead;
import org.simantics.utils.Development;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final public class AsyncReadEntry<T> extends CacheEntryBase<AsyncProcedure<T>> implements AsyncProcedure<T>, IPending {

    private static final Logger LOGGER = LoggerFactory.getLogger(AsyncReadEntry.class);

    protected AsyncRead<T> id;
    protected PendingTaskSupport pendingTaskSupport;

    AsyncReadEntry(AsyncRead<T> request) {
        this.id = request;
		if (Development.DEVELOPMENT) {
			if(Development.<Boolean>getProperty(DevelopmentKeys.CACHE_ENTRY_STATE, Bindings.BOOLEAN)) {
		        System.err.println("[QUERY STATE]: created " + this);
			}
		}
    }

    @Override
    int makeHash() {
        return id.hashCode();
    }

    @Override
    public Object getOriginalRequest() {
        return id;
    }

    @Override
    public void discard() {
        super.discard();
        setResult(null);
    }

    public void except(AsyncReadGraph graph, Throwable t) {

        assert (isPending());

        synchronized (this) {
            except(t);
        }

    }

    @Override
    final public Query getQuery() {

        return new Query() {

            @Override
            public void recompute(ReadGraphImpl graph) {

                try {

                    BlockingAsyncProcedure<T> proc = new BlockingAsyncProcedure(graph, AsyncReadEntry.this, new AsyncProcedure<T>() {

                        @Override
                        public void execute(AsyncReadGraph graph, T result) {
                            setResult(result);
                            setReady();
                        }

                        @Override
                        public void exception(AsyncReadGraph graph, Throwable t) {
                            except(t);
                        }

                    }, id, true);
                    
                    proc.performSync(id);

                } catch (Throwable t) {
                    except(t);
                }

            }

            @Override
            public void removeEntry(QueryProcessor qp) {
                qp.cache.remove(AsyncReadEntry.this);
            }

            @Override
            public int type() {
                return id.getFlags();
            }

            @Override
            public String toString() {
                if (id == null)
                    return "DISCARDED";
                else if (isExcepted())
                    return id.toString() + " " + getResult();
                else
                    return id.toString() + " " + statusOrException;
            }

        };

    }

    @Override
    public Object performFromCache(ReadGraphImpl graph, AsyncProcedure<T> proc) {

        if (isExcepted()) {

            try {
                proc.exception(graph, (Throwable) getResult());
            } catch (Throwable t) {
                LOGGER.error("performFromCache proc.exception failed", t);
            }

        } else {

            try {
                T result = (T) getResult();
                proc.execute(graph, result);
            } catch (Throwable t) {
                LOGGER.error("performFromCache proc.execute failed", t);
            }

        }

        return getResult();

    }

    public static <T> T computeForEach(ReadGraphImpl callerGraph, AsyncRead<T> request, AsyncReadEntry<T> entry,
            AsyncProcedure<T> procedure_, boolean needsToBlock) throws DatabaseException {

        BlockingAsyncProcedure<T> proc = new BlockingAsyncProcedure(callerGraph, entry, procedure_, request, needsToBlock);
        if(needsToBlock) {
            return proc.performSync(request);
        } else {
            proc.performAsync(request);
            return null;
        }

    }

    @Override
    public String toString() {
        if (isDiscarded())
            return "DISCARDED " + id.toString();
        else if (isExcepted())
            return id.toString() + " " + getResult();
        else
            return id.toString() + " " + statusOrException;
    }

    @Override
    public void execute(AsyncReadGraph graph, T result) {
        Collection<SessionTask> tasks = null;
        synchronized(this) {
            setResult(result);
            setReady();
            if(pendingTaskSupport != null)
                tasks = pendingTaskSupport.executePending();
        }
        if(tasks != null)
            for(SessionTask task : tasks)
                ((ReadGraphImpl)graph).processor.scheduleNow(task);
    }

    @Override
    public synchronized void exception(AsyncReadGraph graph, Throwable throwable) {
        Collection<SessionTask> tasks = null;
        synchronized(this) {
            except(throwable);
            if(pendingTaskSupport != null)
                tasks = pendingTaskSupport.executePending();
        }
        if(tasks != null)
            for(SessionTask task : tasks)
                ((ReadGraphImpl)graph).processor.scheduleNow(task);
    }

    public void executeWhenResultIsAvailable(QueryProcessor processor, SessionTask task) {
        boolean ready = false;
        synchronized(this) {
            if(pendingTaskSupport == null)
                pendingTaskSupport = new PendingTaskSupport(this);
            ready = pendingTaskSupport.executeWhenResultIsAvailable(task);
        }
        if(ready) {
            processor.scheduleNow(task);
        }
    }

}
