package org.simantics.databoard.serialization.impl;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.util.List;

import org.simantics.databoard.binding.ArrayBinding;
import org.simantics.databoard.serialization.SerializationException;
import org.simantics.databoard.serialization.Serializer.NonRecursiveSerializer;
import org.simantics.databoard.type.ArrayType;
import org.simantics.databoard.util.Range;

public class BooleanArraySerializer extends NonRecursiveSerializer {

	Range length;
	Integer fixedLength, fixedSize;
	
	public BooleanArraySerializer(ArrayBinding binding)
	{
		ArrayType arrayType = (ArrayType) binding.type();
		this.length = arrayType.getLength();
		if (length!=null && length.getLower().equals(length.getUpper()) && length.getLower().getValue()!=null)
		{
			fixedLength = length.getLower().getValue().intValue();
			fixedSize = fixedLength * 1;
		}
	}
	
	@Override
	public Object deserialize(DataInput in)
	throws IOException, SerializationException 
	{
		int length = fixedLength != null ? fixedLength : in.readInt();
		if (length<0) throw new SerializationException("Cannot use negative array length");
		assertRemainingBytes(in, length);			
		
		boolean[] array = new boolean[length];
		for(int i=0;i<array.length;++i) {
			byte value = in.readByte();
			if (value==0) array[i] = false;
			else if (value==1) array[i] = true;
			else throw new SerializationException("Unexpected value \""+value+"\" for boolean");
		}
		return array;
	}
	
	public Object deserializeToTry(DataInput in, List<Object> identities, Object obj) throws IOException
	{
		int length = fixedLength != null ? fixedLength : in.readInt();
		boolean[] array = (boolean[]) obj;
		if (length!=array.length) array = new boolean[ length ];
		assertRemainingBytes(in, length);			
		
		for (int i=0; i<array.length;i++) {
			byte value = in.readByte();
			if (value==0) array[i] = false;
			else if (value==1) array[i] = true;
			else throw new SerializationException("Unexpected value \""+value+"\" for boolean");
		}
		
		return array;
	}

	@Override
	public void deserializeTo(DataInput in, Object obj)
			throws IOException { 
		int length = fixedLength != null ? fixedLength : in.readInt();
		boolean[] array = (boolean[]) obj;
		if (length!=array.length) throw new SerializationException("primitive array is size immutable");
		assertRemainingBytes(in, length);			
		
		for (int i=0; i<array.length;i++) {
			byte value = in.readByte();
			if (value==0) array[i] = false;
			else if (value==1) array[i] = true;
			else throw new SerializationException("Unexpected value \""+value+"\" for boolean");
		}
	}

	@Override
	public void skip(DataInput in)
			throws IOException {
		int length = fixedSize != null ? fixedLength : in.readInt();			
		in.skipBytes(length);
	}
	
	@Override
	public void serialize(DataOutput out, Object obj)
	throws IOException 
	{
		boolean[] array = (boolean[])obj;
		if (fixedLength==null) 
			out.writeInt(array.length);
		for(boolean f : array)
			out.write( (byte) (f ? 1 : 0) );
	}

	@Override
	public Integer getConstantSize() {
		return fixedSize;
	}

	@Override
	public int getSize(Object obj) {
		if (fixedSize!=null) return fixedSize;
		boolean[] array = (boolean[])obj;			
		return 4 + array.length;
	}

	@Override
	public int getMinSize() {
		return fixedSize != null ? fixedSize : 4;
	}
	
}