/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.koloboke.collect.map.hash;

import com.koloboke.collect.map.CharObjMap;
import com.koloboke.compile.KolobokeMap;
import java.util.function.Consumer;

import javax.annotation.Nonnull;
import java.util.Map;
import java.util.ServiceLoader;


/**
 * This class consists only of static factory methods to construct {@code HashCharObjMap}s, and
 * the default {@link HashCharObjMapFactory} static provider ({@link #getDefaultFactory()}).
 *
 * @see HashCharObjMap
 * @see KolobokeMap @KolobokeMap
 */
public final class HashCharObjMaps {
    
    private static class DefaultFactoryHolder {
        private static final HashCharObjMapFactory defaultFactory =
                ServiceLoader.load(HashCharObjMapFactory.class).iterator().next();
    }

    /**
     * Returns the default {@link HashCharObjMapFactory} implementation, to which
     * all static methods in this class delegate.
     *
     
      * @param <V> the most general value type of the maps that could
                               be constructed by the returned factory 
     * @return the default {@link HashCharObjMapFactory} implementation
     * @throws RuntimeException if no implementations
     *         of {@link HashCharObjMapFactory} are provided
     */
    @Nonnull
    public static <V> HashCharObjMapFactory<V> getDefaultFactory() {
        return (HashCharObjMapFactory<V>) DefaultFactoryHolder.defaultFactory;
    }

    

    

    

    


    /**
     * Constructs a new empty mutable map of the default expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newMutableMap() newMutableMap()}.
     *
     
      * @param <V> the value type of the returned map 
     * @return a new empty mutable map
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newMutableMap() {
        return getDefaultFactory().<V>newMutableMap();
    }

    /**
     * Constructs a new empty mutable map of the given expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newMutableMap(int) newMutableMap(expectedSize)}.
     *
     * @param expectedSize the expected size of the returned map
    
      * @param <V> the value type of the returned map 
     * @return a new empty mutable map of the given expected size
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newMutableMap(int expectedSize) {
        return getDefaultFactory().<V>newMutableMap(expectedSize);
    }

    
    
    
    


    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newMutableMap(Map,
     * Map, int) newMutableMap(map1, map2, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * @param expectedSize the expected size of the returned map
    
      * @param <V> the value type of the returned map 
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newMutableMap(
            @Nonnull Map<Character, ? extends V> map1,
            @Nonnull Map<Character, ? extends V> map2, int expectedSize) {
        return getDefaultFactory().<V>newMutableMap(map1, map2, expectedSize);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newMutableMap(Map, Map,
     * Map, int) newMutableMap(map1, map2, map3, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * @param expectedSize the expected size of the returned map
    
      * @param <V> the value type of the returned map 
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newMutableMap(
            @Nonnull Map<Character, ? extends V> map1,
            @Nonnull Map<Character, ? extends V> map2,
            @Nonnull Map<Character, ? extends V> map3, int expectedSize) {
        return getDefaultFactory().<V>newMutableMap(map1, map2, map3, expectedSize);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newMutableMap(Map, Map,
     * Map, Map, int) newMutableMap(map1, map2, map3, map4, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * @param expectedSize the expected size of the returned map
    
      * @param <V> the value type of the returned map 
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newMutableMap(
            @Nonnull Map<Character, ? extends V> map1,
            @Nonnull Map<Character, ? extends V> map2,
            @Nonnull Map<Character, ? extends V> map3,
            @Nonnull Map<Character, ? extends V> map4, int expectedSize) {
        return getDefaultFactory().<V>newMutableMap(map1, map2, map3, map4, expectedSize);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newMutableMap(Map, Map,
     * Map, Map, Map, int) newMutableMap(map1, map2, map3, map4, map5, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * @param expectedSize the expected size of the returned map
    
      * @param <V> the value type of the returned map 
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newMutableMap(
            @Nonnull Map<Character, ? extends V> map1,
            @Nonnull Map<Character, ? extends V> map2,
            @Nonnull Map<Character, ? extends V> map3,
            @Nonnull Map<Character, ? extends V> map4,
            @Nonnull Map<Character, ? extends V> map5, int expectedSize) {
        return getDefaultFactory().<V>newMutableMap(map1, map2, map3, map4, map5, expectedSize);
    }

    /**
     * Constructs a new mutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newMutableMap(
     * Consumer, int) newMutableMap(entriesSupplier, expectedSize)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * @param expectedSize the expected size of the returned map
    
      * @param <V> the value type of the returned map 
     * @return a new mutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newMutableMap(@Nonnull
            Consumer<com.koloboke.function.CharObjConsumer<V>> entriesSupplier
            , int expectedSize) {
        return getDefaultFactory().<V>newMutableMap(entriesSupplier, expectedSize);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newMutableMap(char[], Object[], int
     * ) newMutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
    
      * @param <V> the value type of the returned map 
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newMutableMap(
            @Nonnull char[] keys, @Nonnull V[] values, int expectedSize) {
        return getDefaultFactory().<V>newMutableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newMutableMap(Character[],
     * Object[], int) newMutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
    
      * @param <V> the value type of the returned map 
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newMutableMap(
            @Nonnull Character[] keys, @Nonnull V[] values, int expectedSize) {
        return getDefaultFactory().<V>newMutableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newMutableMap(Iterable, Iterable, int
     * ) newMutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * @param expectedSize the expected size of the returned map
    
      * @param <V> the value type of the returned map 
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newMutableMap(
            @Nonnull Iterable<Character> keys, @Nonnull Iterable<? extends V> values
            , int expectedSize) {
        return getDefaultFactory().<V>newMutableMap(keys, values, expectedSize);
    }
    
    
    
    

    /**
     * Constructs a new mutable map with the same mappings as the specified {@code map}.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newMutableMap(
     * Map) newMutableMap(map)}.
     *
     * @param map the map whose mappings are to be placed in the returned map
     * 
    
      * @param <V> the value type of the returned map 
     * @return a new mutable map with the same mappings as the specified {@code map}
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newMutableMap(
            @Nonnull Map<Character, ? extends V> map) {
        return getDefaultFactory().<V>newMutableMap(map);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newMutableMap(Map,
     * Map) newMutableMap(map1, map2)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * 
    
      * @param <V> the value type of the returned map 
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newMutableMap(
            @Nonnull Map<Character, ? extends V> map1,
            @Nonnull Map<Character, ? extends V> map2) {
        return getDefaultFactory().<V>newMutableMap(map1, map2);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newMutableMap(Map, Map,
     * Map) newMutableMap(map1, map2, map3)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * 
    
      * @param <V> the value type of the returned map 
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newMutableMap(
            @Nonnull Map<Character, ? extends V> map1,
            @Nonnull Map<Character, ? extends V> map2,
            @Nonnull Map<Character, ? extends V> map3) {
        return getDefaultFactory().<V>newMutableMap(map1, map2, map3);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newMutableMap(Map, Map,
     * Map, Map) newMutableMap(map1, map2, map3, map4)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * 
    
      * @param <V> the value type of the returned map 
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newMutableMap(
            @Nonnull Map<Character, ? extends V> map1,
            @Nonnull Map<Character, ? extends V> map2,
            @Nonnull Map<Character, ? extends V> map3,
            @Nonnull Map<Character, ? extends V> map4) {
        return getDefaultFactory().<V>newMutableMap(map1, map2, map3, map4);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newMutableMap(Map, Map,
     * Map, Map, Map) newMutableMap(map1, map2, map3, map4, map5)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * 
    
      * @param <V> the value type of the returned map 
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newMutableMap(
            @Nonnull Map<Character, ? extends V> map1,
            @Nonnull Map<Character, ? extends V> map2,
            @Nonnull Map<Character, ? extends V> map3,
            @Nonnull Map<Character, ? extends V> map4,
            @Nonnull Map<Character, ? extends V> map5) {
        return getDefaultFactory().<V>newMutableMap(map1, map2, map3, map4, map5);
    }

    /**
     * Constructs a new mutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newMutableMap(
     * Consumer) newMutableMap(entriesSupplier)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * 
    
      * @param <V> the value type of the returned map 
     * @return a new mutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newMutableMap(@Nonnull
            Consumer<com.koloboke.function.CharObjConsumer<V>> entriesSupplier
            ) {
        return getDefaultFactory().<V>newMutableMap(entriesSupplier);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newMutableMap(char[], Object[]
     * ) newMutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
    
      * @param <V> the value type of the returned map 
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newMutableMap(
            @Nonnull char[] keys, @Nonnull V[] values) {
        return getDefaultFactory().<V>newMutableMap(keys, values);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newMutableMap(Character[],
     * Object[]) newMutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
    
      * @param <V> the value type of the returned map 
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newMutableMap(
            @Nonnull Character[] keys, @Nonnull V[] values) {
        return getDefaultFactory().<V>newMutableMap(keys, values);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newMutableMap(Iterable, Iterable
     * ) newMutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * 
    
      * @param <V> the value type of the returned map 
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newMutableMap(
            @Nonnull Iterable<Character> keys, @Nonnull Iterable<? extends V> values
            ) {
        return getDefaultFactory().<V>newMutableMap(keys, values);
    }

    /**
     * Constructs a new mutable map of the single specified mapping.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newMutableMapOf(char, Object
     * ) newMutableMapOf(k1, v1)}.
     *
     * @param k1 the key of the sole mapping
     * @param v1 the value of the sole mapping
    
      * @param <V> the value type of the returned map 
     * @return a new mutable map of the single specified mapping
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newMutableMapOf(
            char k1, V v1) {
        return getDefaultFactory().<V>newMutableMapOf(k1, v1);
    }

    /**
     * Constructs a new mutable map of the two specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newMutableMapOf(char, Object,
     * char, Object) newMutableMapOf(k1, v1, k2, v2)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
    
      * @param <V> the value type of the returned map 
     * @return a new mutable map of the two specified mappings
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newMutableMapOf(
            char k1, V v1, char k2, V v2) {
        return getDefaultFactory().<V>newMutableMapOf(k1, v1, k2, v2);
    }

    /**
     * Constructs a new mutable map of the three specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newMutableMapOf(char, Object,
     * char, Object, char, Object) newMutableMapOf(k1, v1, k2, v2,
     * k3, v3)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
    
      * @param <V> the value type of the returned map 
     * @return a new mutable map of the three specified mappings
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newMutableMapOf(
            char k1, V v1, char k2, V v2,
            char k3, V v3) {
        return getDefaultFactory().<V>newMutableMapOf(k1, v1, k2, v2, k3, v3);
    }

    /**
     * Constructs a new mutable map of the four specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newMutableMapOf(char, Object,
     * char, Object, char, Object, char, Object
     * ) newMutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
    
      * @param <V> the value type of the returned map 
     * @return a new mutable map of the four specified mappings
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newMutableMapOf(
            char k1, V v1, char k2, V v2,
            char k3, V v3, char k4, V v4) {
        return getDefaultFactory().<V>newMutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4);
    }

    /**
     * Constructs a new mutable map of the five specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newMutableMapOf(char, Object,
     * char, Object, char, Object, char, Object,
     * char, Object) newMutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param k5 the key of the fifth mapping
     * @param v5 the value of the fifth mapping
    
      * @param <V> the value type of the returned map 
     * @return a new mutable map of the five specified mappings
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newMutableMapOf(
            char k1, V v1, char k2, V v2,
            char k3, V v3, char k4, V v4,
            char k5, V v5) {
        return getDefaultFactory().<V>newMutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5);
    }

    /**
     * Constructs a new empty updatable map of the default expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newUpdatableMap() newUpdatableMap()}.
     *
     
      * @param <V> the value type of the returned map 
     * @return a new empty updatable map
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newUpdatableMap() {
        return getDefaultFactory().<V>newUpdatableMap();
    }

    /**
     * Constructs a new empty updatable map of the given expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newUpdatableMap(int) newUpdatableMap(expectedSize)}.
     *
     * @param expectedSize the expected size of the returned map
    
      * @param <V> the value type of the returned map 
     * @return a new empty updatable map of the given expected size
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newUpdatableMap(int expectedSize) {
        return getDefaultFactory().<V>newUpdatableMap(expectedSize);
    }

    
    
    
    


    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newUpdatableMap(Map,
     * Map, int) newUpdatableMap(map1, map2, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * @param expectedSize the expected size of the returned map
    
      * @param <V> the value type of the returned map 
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newUpdatableMap(
            @Nonnull Map<Character, ? extends V> map1,
            @Nonnull Map<Character, ? extends V> map2, int expectedSize) {
        return getDefaultFactory().<V>newUpdatableMap(map1, map2, expectedSize);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newUpdatableMap(Map, Map,
     * Map, int) newUpdatableMap(map1, map2, map3, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * @param expectedSize the expected size of the returned map
    
      * @param <V> the value type of the returned map 
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newUpdatableMap(
            @Nonnull Map<Character, ? extends V> map1,
            @Nonnull Map<Character, ? extends V> map2,
            @Nonnull Map<Character, ? extends V> map3, int expectedSize) {
        return getDefaultFactory().<V>newUpdatableMap(map1, map2, map3, expectedSize);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newUpdatableMap(Map, Map,
     * Map, Map, int) newUpdatableMap(map1, map2, map3, map4, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * @param expectedSize the expected size of the returned map
    
      * @param <V> the value type of the returned map 
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newUpdatableMap(
            @Nonnull Map<Character, ? extends V> map1,
            @Nonnull Map<Character, ? extends V> map2,
            @Nonnull Map<Character, ? extends V> map3,
            @Nonnull Map<Character, ? extends V> map4, int expectedSize) {
        return getDefaultFactory().<V>newUpdatableMap(map1, map2, map3, map4, expectedSize);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newUpdatableMap(Map, Map,
     * Map, Map, Map, int) newUpdatableMap(map1, map2, map3, map4, map5, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * @param expectedSize the expected size of the returned map
    
      * @param <V> the value type of the returned map 
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newUpdatableMap(
            @Nonnull Map<Character, ? extends V> map1,
            @Nonnull Map<Character, ? extends V> map2,
            @Nonnull Map<Character, ? extends V> map3,
            @Nonnull Map<Character, ? extends V> map4,
            @Nonnull Map<Character, ? extends V> map5, int expectedSize) {
        return getDefaultFactory().<V>newUpdatableMap(map1, map2, map3, map4, map5, expectedSize);
    }

    /**
     * Constructs a new updatable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newUpdatableMap(
     * Consumer, int) newUpdatableMap(entriesSupplier, expectedSize)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * @param expectedSize the expected size of the returned map
    
      * @param <V> the value type of the returned map 
     * @return a new updatable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newUpdatableMap(@Nonnull
            Consumer<com.koloboke.function.CharObjConsumer<V>> entriesSupplier
            , int expectedSize) {
        return getDefaultFactory().<V>newUpdatableMap(entriesSupplier, expectedSize);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newUpdatableMap(char[], Object[], int
     * ) newUpdatableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
    
      * @param <V> the value type of the returned map 
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newUpdatableMap(
            @Nonnull char[] keys, @Nonnull V[] values, int expectedSize) {
        return getDefaultFactory().<V>newUpdatableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newUpdatableMap(Character[],
     * Object[], int) newUpdatableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
    
      * @param <V> the value type of the returned map 
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newUpdatableMap(
            @Nonnull Character[] keys, @Nonnull V[] values, int expectedSize) {
        return getDefaultFactory().<V>newUpdatableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newUpdatableMap(Iterable, Iterable, int
     * ) newUpdatableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * @param expectedSize the expected size of the returned map
    
      * @param <V> the value type of the returned map 
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newUpdatableMap(
            @Nonnull Iterable<Character> keys, @Nonnull Iterable<? extends V> values
            , int expectedSize) {
        return getDefaultFactory().<V>newUpdatableMap(keys, values, expectedSize);
    }
    
    
    
    

    /**
     * Constructs a new updatable map with the same mappings as the specified {@code map}.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newUpdatableMap(
     * Map) newUpdatableMap(map)}.
     *
     * @param map the map whose mappings are to be placed in the returned map
     * 
    
      * @param <V> the value type of the returned map 
     * @return a new updatable map with the same mappings as the specified {@code map}
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newUpdatableMap(
            @Nonnull Map<Character, ? extends V> map) {
        return getDefaultFactory().<V>newUpdatableMap(map);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newUpdatableMap(Map,
     * Map) newUpdatableMap(map1, map2)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * 
    
      * @param <V> the value type of the returned map 
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newUpdatableMap(
            @Nonnull Map<Character, ? extends V> map1,
            @Nonnull Map<Character, ? extends V> map2) {
        return getDefaultFactory().<V>newUpdatableMap(map1, map2);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newUpdatableMap(Map, Map,
     * Map) newUpdatableMap(map1, map2, map3)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * 
    
      * @param <V> the value type of the returned map 
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newUpdatableMap(
            @Nonnull Map<Character, ? extends V> map1,
            @Nonnull Map<Character, ? extends V> map2,
            @Nonnull Map<Character, ? extends V> map3) {
        return getDefaultFactory().<V>newUpdatableMap(map1, map2, map3);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newUpdatableMap(Map, Map,
     * Map, Map) newUpdatableMap(map1, map2, map3, map4)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * 
    
      * @param <V> the value type of the returned map 
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newUpdatableMap(
            @Nonnull Map<Character, ? extends V> map1,
            @Nonnull Map<Character, ? extends V> map2,
            @Nonnull Map<Character, ? extends V> map3,
            @Nonnull Map<Character, ? extends V> map4) {
        return getDefaultFactory().<V>newUpdatableMap(map1, map2, map3, map4);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newUpdatableMap(Map, Map,
     * Map, Map, Map) newUpdatableMap(map1, map2, map3, map4, map5)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * 
    
      * @param <V> the value type of the returned map 
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newUpdatableMap(
            @Nonnull Map<Character, ? extends V> map1,
            @Nonnull Map<Character, ? extends V> map2,
            @Nonnull Map<Character, ? extends V> map3,
            @Nonnull Map<Character, ? extends V> map4,
            @Nonnull Map<Character, ? extends V> map5) {
        return getDefaultFactory().<V>newUpdatableMap(map1, map2, map3, map4, map5);
    }

    /**
     * Constructs a new updatable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newUpdatableMap(
     * Consumer) newUpdatableMap(entriesSupplier)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * 
    
      * @param <V> the value type of the returned map 
     * @return a new updatable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newUpdatableMap(@Nonnull
            Consumer<com.koloboke.function.CharObjConsumer<V>> entriesSupplier
            ) {
        return getDefaultFactory().<V>newUpdatableMap(entriesSupplier);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newUpdatableMap(char[], Object[]
     * ) newUpdatableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
    
      * @param <V> the value type of the returned map 
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newUpdatableMap(
            @Nonnull char[] keys, @Nonnull V[] values) {
        return getDefaultFactory().<V>newUpdatableMap(keys, values);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newUpdatableMap(Character[],
     * Object[]) newUpdatableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
    
      * @param <V> the value type of the returned map 
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newUpdatableMap(
            @Nonnull Character[] keys, @Nonnull V[] values) {
        return getDefaultFactory().<V>newUpdatableMap(keys, values);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newUpdatableMap(Iterable, Iterable
     * ) newUpdatableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * 
    
      * @param <V> the value type of the returned map 
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newUpdatableMap(
            @Nonnull Iterable<Character> keys, @Nonnull Iterable<? extends V> values
            ) {
        return getDefaultFactory().<V>newUpdatableMap(keys, values);
    }

    /**
     * Constructs a new updatable map of the single specified mapping.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newUpdatableMapOf(char, Object
     * ) newUpdatableMapOf(k1, v1)}.
     *
     * @param k1 the key of the sole mapping
     * @param v1 the value of the sole mapping
    
      * @param <V> the value type of the returned map 
     * @return a new updatable map of the single specified mapping
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newUpdatableMapOf(
            char k1, V v1) {
        return getDefaultFactory().<V>newUpdatableMapOf(k1, v1);
    }

    /**
     * Constructs a new updatable map of the two specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newUpdatableMapOf(char, Object,
     * char, Object) newUpdatableMapOf(k1, v1, k2, v2)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
    
      * @param <V> the value type of the returned map 
     * @return a new updatable map of the two specified mappings
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newUpdatableMapOf(
            char k1, V v1, char k2, V v2) {
        return getDefaultFactory().<V>newUpdatableMapOf(k1, v1, k2, v2);
    }

    /**
     * Constructs a new updatable map of the three specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newUpdatableMapOf(char, Object,
     * char, Object, char, Object) newUpdatableMapOf(k1, v1, k2, v2,
     * k3, v3)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
    
      * @param <V> the value type of the returned map 
     * @return a new updatable map of the three specified mappings
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newUpdatableMapOf(
            char k1, V v1, char k2, V v2,
            char k3, V v3) {
        return getDefaultFactory().<V>newUpdatableMapOf(k1, v1, k2, v2, k3, v3);
    }

    /**
     * Constructs a new updatable map of the four specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newUpdatableMapOf(char, Object,
     * char, Object, char, Object, char, Object
     * ) newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
    
      * @param <V> the value type of the returned map 
     * @return a new updatable map of the four specified mappings
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newUpdatableMapOf(
            char k1, V v1, char k2, V v2,
            char k3, V v3, char k4, V v4) {
        return getDefaultFactory().<V>newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4);
    }

    /**
     * Constructs a new updatable map of the five specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newUpdatableMapOf(char, Object,
     * char, Object, char, Object, char, Object,
     * char, Object) newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param k5 the key of the fifth mapping
     * @param v5 the value of the fifth mapping
    
      * @param <V> the value type of the returned map 
     * @return a new updatable map of the five specified mappings
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newUpdatableMapOf(
            char k1, V v1, char k2, V v2,
            char k3, V v3, char k4, V v4,
            char k5, V v5) {
        return getDefaultFactory().<V>newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5);
    }


    
    
    
    


    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newImmutableMap(Map,
     * Map, int) newImmutableMap(map1, map2, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * @param expectedSize the expected size of the returned map
    
      * @param <V> the value type of the returned map 
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newImmutableMap(
            @Nonnull Map<Character, ? extends V> map1,
            @Nonnull Map<Character, ? extends V> map2, int expectedSize) {
        return getDefaultFactory().<V>newImmutableMap(map1, map2, expectedSize);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newImmutableMap(Map, Map,
     * Map, int) newImmutableMap(map1, map2, map3, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * @param expectedSize the expected size of the returned map
    
      * @param <V> the value type of the returned map 
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newImmutableMap(
            @Nonnull Map<Character, ? extends V> map1,
            @Nonnull Map<Character, ? extends V> map2,
            @Nonnull Map<Character, ? extends V> map3, int expectedSize) {
        return getDefaultFactory().<V>newImmutableMap(map1, map2, map3, expectedSize);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newImmutableMap(Map, Map,
     * Map, Map, int) newImmutableMap(map1, map2, map3, map4, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * @param expectedSize the expected size of the returned map
    
      * @param <V> the value type of the returned map 
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newImmutableMap(
            @Nonnull Map<Character, ? extends V> map1,
            @Nonnull Map<Character, ? extends V> map2,
            @Nonnull Map<Character, ? extends V> map3,
            @Nonnull Map<Character, ? extends V> map4, int expectedSize) {
        return getDefaultFactory().<V>newImmutableMap(map1, map2, map3, map4, expectedSize);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newImmutableMap(Map, Map,
     * Map, Map, Map, int) newImmutableMap(map1, map2, map3, map4, map5, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * @param expectedSize the expected size of the returned map
    
      * @param <V> the value type of the returned map 
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newImmutableMap(
            @Nonnull Map<Character, ? extends V> map1,
            @Nonnull Map<Character, ? extends V> map2,
            @Nonnull Map<Character, ? extends V> map3,
            @Nonnull Map<Character, ? extends V> map4,
            @Nonnull Map<Character, ? extends V> map5, int expectedSize) {
        return getDefaultFactory().<V>newImmutableMap(map1, map2, map3, map4, map5, expectedSize);
    }

    /**
     * Constructs a new immutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newImmutableMap(
     * Consumer, int) newImmutableMap(entriesSupplier, expectedSize)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * @param expectedSize the expected size of the returned map
    
      * @param <V> the value type of the returned map 
     * @return a new immutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newImmutableMap(@Nonnull
            Consumer<com.koloboke.function.CharObjConsumer<V>> entriesSupplier
            , int expectedSize) {
        return getDefaultFactory().<V>newImmutableMap(entriesSupplier, expectedSize);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newImmutableMap(char[], Object[], int
     * ) newImmutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
    
      * @param <V> the value type of the returned map 
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newImmutableMap(
            @Nonnull char[] keys, @Nonnull V[] values, int expectedSize) {
        return getDefaultFactory().<V>newImmutableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newImmutableMap(Character[],
     * Object[], int) newImmutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
    
      * @param <V> the value type of the returned map 
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newImmutableMap(
            @Nonnull Character[] keys, @Nonnull V[] values, int expectedSize) {
        return getDefaultFactory().<V>newImmutableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newImmutableMap(Iterable, Iterable, int
     * ) newImmutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * @param expectedSize the expected size of the returned map
    
      * @param <V> the value type of the returned map 
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newImmutableMap(
            @Nonnull Iterable<Character> keys, @Nonnull Iterable<? extends V> values
            , int expectedSize) {
        return getDefaultFactory().<V>newImmutableMap(keys, values, expectedSize);
    }
    
    
    
    

    /**
     * Constructs a new immutable map with the same mappings as the specified {@code map}.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newImmutableMap(
     * Map) newImmutableMap(map)}.
     *
     * @param map the map whose mappings are to be placed in the returned map
     * 
    
      * @param <V> the value type of the returned map 
     * @return a new immutable map with the same mappings as the specified {@code map}
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newImmutableMap(
            @Nonnull Map<Character, ? extends V> map) {
        return getDefaultFactory().<V>newImmutableMap(map);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newImmutableMap(Map,
     * Map) newImmutableMap(map1, map2)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * 
    
      * @param <V> the value type of the returned map 
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newImmutableMap(
            @Nonnull Map<Character, ? extends V> map1,
            @Nonnull Map<Character, ? extends V> map2) {
        return getDefaultFactory().<V>newImmutableMap(map1, map2);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newImmutableMap(Map, Map,
     * Map) newImmutableMap(map1, map2, map3)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * 
    
      * @param <V> the value type of the returned map 
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newImmutableMap(
            @Nonnull Map<Character, ? extends V> map1,
            @Nonnull Map<Character, ? extends V> map2,
            @Nonnull Map<Character, ? extends V> map3) {
        return getDefaultFactory().<V>newImmutableMap(map1, map2, map3);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newImmutableMap(Map, Map,
     * Map, Map) newImmutableMap(map1, map2, map3, map4)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * 
    
      * @param <V> the value type of the returned map 
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newImmutableMap(
            @Nonnull Map<Character, ? extends V> map1,
            @Nonnull Map<Character, ? extends V> map2,
            @Nonnull Map<Character, ? extends V> map3,
            @Nonnull Map<Character, ? extends V> map4) {
        return getDefaultFactory().<V>newImmutableMap(map1, map2, map3, map4);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newImmutableMap(Map, Map,
     * Map, Map, Map) newImmutableMap(map1, map2, map3, map4, map5)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * 
    
      * @param <V> the value type of the returned map 
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newImmutableMap(
            @Nonnull Map<Character, ? extends V> map1,
            @Nonnull Map<Character, ? extends V> map2,
            @Nonnull Map<Character, ? extends V> map3,
            @Nonnull Map<Character, ? extends V> map4,
            @Nonnull Map<Character, ? extends V> map5) {
        return getDefaultFactory().<V>newImmutableMap(map1, map2, map3, map4, map5);
    }

    /**
     * Constructs a new immutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newImmutableMap(
     * Consumer) newImmutableMap(entriesSupplier)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * 
    
      * @param <V> the value type of the returned map 
     * @return a new immutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newImmutableMap(@Nonnull
            Consumer<com.koloboke.function.CharObjConsumer<V>> entriesSupplier
            ) {
        return getDefaultFactory().<V>newImmutableMap(entriesSupplier);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newImmutableMap(char[], Object[]
     * ) newImmutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
    
      * @param <V> the value type of the returned map 
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newImmutableMap(
            @Nonnull char[] keys, @Nonnull V[] values) {
        return getDefaultFactory().<V>newImmutableMap(keys, values);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newImmutableMap(Character[],
     * Object[]) newImmutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
    
      * @param <V> the value type of the returned map 
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newImmutableMap(
            @Nonnull Character[] keys, @Nonnull V[] values) {
        return getDefaultFactory().<V>newImmutableMap(keys, values);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newImmutableMap(Iterable, Iterable
     * ) newImmutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * 
    
      * @param <V> the value type of the returned map 
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newImmutableMap(
            @Nonnull Iterable<Character> keys, @Nonnull Iterable<? extends V> values
            ) {
        return getDefaultFactory().<V>newImmutableMap(keys, values);
    }

    /**
     * Constructs a new immutable map of the single specified mapping.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newImmutableMapOf(char, Object
     * ) newImmutableMapOf(k1, v1)}.
     *
     * @param k1 the key of the sole mapping
     * @param v1 the value of the sole mapping
    
      * @param <V> the value type of the returned map 
     * @return a new immutable map of the single specified mapping
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newImmutableMapOf(
            char k1, V v1) {
        return getDefaultFactory().<V>newImmutableMapOf(k1, v1);
    }

    /**
     * Constructs a new immutable map of the two specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newImmutableMapOf(char, Object,
     * char, Object) newImmutableMapOf(k1, v1, k2, v2)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
    
      * @param <V> the value type of the returned map 
     * @return a new immutable map of the two specified mappings
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newImmutableMapOf(
            char k1, V v1, char k2, V v2) {
        return getDefaultFactory().<V>newImmutableMapOf(k1, v1, k2, v2);
    }

    /**
     * Constructs a new immutable map of the three specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newImmutableMapOf(char, Object,
     * char, Object, char, Object) newImmutableMapOf(k1, v1, k2, v2,
     * k3, v3)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
    
      * @param <V> the value type of the returned map 
     * @return a new immutable map of the three specified mappings
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newImmutableMapOf(
            char k1, V v1, char k2, V v2,
            char k3, V v3) {
        return getDefaultFactory().<V>newImmutableMapOf(k1, v1, k2, v2, k3, v3);
    }

    /**
     * Constructs a new immutable map of the four specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newImmutableMapOf(char, Object,
     * char, Object, char, Object, char, Object
     * ) newImmutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
    
      * @param <V> the value type of the returned map 
     * @return a new immutable map of the four specified mappings
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newImmutableMapOf(
            char k1, V v1, char k2, V v2,
            char k3, V v3, char k4, V v4) {
        return getDefaultFactory().<V>newImmutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4);
    }

    /**
     * Constructs a new immutable map of the five specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharObjMapFactory#newImmutableMapOf(char, Object,
     * char, Object, char, Object, char, Object,
     * char, Object) newImmutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param k5 the key of the fifth mapping
     * @param v5 the value of the fifth mapping
    
      * @param <V> the value type of the returned map 
     * @return a new immutable map of the five specified mappings
     */
    @Nonnull
    public static <V> HashCharObjMap<V> newImmutableMapOf(
            char k1, V v1, char k2, V v2,
            char k3, V v3, char k4, V v4,
            char k5, V v5) {
        return getDefaultFactory().<V>newImmutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5);
    }

    private HashCharObjMaps() {}
}
