/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.service;

import java.util.Collection;

import org.simantics.db.AsyncReadGraph;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.procedure.AsyncContextMultiProcedure;
import org.simantics.db.procedure.AsyncMultiProcedure;
import org.simantics.db.request.ExternalRead;
import org.simantics.db.request.Read;


/**
 * @author Antti Villberg
 */
public interface QueryControl {

    interface ControlProcedure {
        void execute(AsyncReadGraph graph);
    }

    /**
     * 
     * @return current amount of queries
     */
    int count();
    
    /**
     * @return The amount of fixed queries after best effort flush
     */
    int flush();
    int flush(ReadGraph graph);
    
    /**
     * 
     * @param allowedTimeInMs The amount of allowed time use in ms.
     */
    void gc(ReadGraph graph, int allowedTimeInMs);

    /**
     * Collects all external read requests and their parent queries that are no
     * longer listened to. Needs to perform exclusive locking of the database
     * client internally so do not invoke this from within a graph read or write
     * request but use {@link #gc(WriteGraph, Collection)} instead.
     * 
     * @param reads
     */
    void gc(Collection<ExternalRead<?>> reads);

    /**
     * Collects all external read requests and their parent queries that are no
     * longer listened to. The write graph argument is used to ensure that the
     * invocation is performed from within a write request with exclusive
     * locking.
     * 
     * @param graph write request handle
     * @param reads the external read requests to attempt garbage collection for
     */
    void gc(WriteGraph graph, Collection<ExternalRead<?>> reads);

    int getAmountOfQueryThreads();

    int getGraphThread(AsyncReadGraph graph);
    
    boolean resume(ReadGraph graph);

    void schedule(AsyncReadGraph graph, int targetThread, ControlProcedure procedure);

    boolean scheduleByCluster(AsyncReadGraph graph, Resource resource, AsyncMultiProcedure<Resource> procedure);
    <C> boolean scheduleByCluster(AsyncReadGraph graph, Resource resource, C context, AsyncContextMultiProcedure<C, Resource> procedure);

    /**
     * Returns a new ReadGraph based on the specified ReadGraph. The returned
     * ReadGraph does not accumulate query dependencies into the requests
     * performed with the specified ReadGraph. DB listeners are therefore not
     * triggered by anything that is performed with the returned ReadGraph.
     * 
     * @Deprecated In favor of syncRequestIndependent 
     * @param graph read transaction handle to clone for listener-independent
     *        use
     * @return read transaction handle that is independent of the requests
     *         performed with the parameter
     */
    ReadGraph getIndependentGraph(ReadGraph graph);
    
    /**
     * Performs the given request without accumulating query dependencies.
     * DB listeners are therefore not triggered by anything that is performed within the request.
     * 
     * @param graph read transaction handle to clone for listener-independent
     *        use
     * @param request the request to perform
     * @return the result of the request evaluation
     */
    <T> T syncRequestIndependent(ReadGraph graph, Read<T> request) throws DatabaseException;
    
    boolean hasParentRequest(ReadGraph graph);

}
